package com.atlassian.user.impl.osuser.config.xml;

/**
 * Utility class which can scan an osuser.xml file and build a {@link DefaultOSUAccessor} implementation
 */

import com.atlassian.cache.CacheFactory;
import com.atlassian.user.cache.CacheFactoryAware;
import com.atlassian.user.configuration.ConfigurationException;
import com.atlassian.user.impl.osuser.DefaultOSUAccessor;
import com.atlassian.user.impl.osuser.OSUAccessor;
import com.opensymphony.user.provider.AccessProvider;
import com.opensymphony.user.provider.CredentialsProvider;
import com.opensymphony.user.provider.ProfileProvider;
import com.opensymphony.user.provider.UserProvider;
import com.opensymphony.util.ClassLoaderUtil;
import org.apache.log4j.Logger;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

public class DefaultOSUConfigurationLoader implements OSUConfigurationLoader
{
    private static final Logger log = Logger.getLogger(DefaultOSUConfigurationLoader.class);
    private DefaultOSUConfigurationHandler configHandler;
    private OSUAccessor accessor = new DefaultOSUAccessor();
    private ProfileProvider profileProvider;
    private AccessProvider accessProvider;
    private CredentialsProvider credentialsProvider;
    private List<CredentialsProvider> credentialProviders = new ArrayList<CredentialsProvider>();
    private CacheFactory cacheFactory;

    public synchronized void load(InputStream in) throws ConfigurationException
    {
        log.debug("Loading config");
        try
        {
            SAXParser parser = SAXParserFactory.newInstance().newSAXParser();
            parser.parse(in, getOSUserConfigurationHandler());
        }
        catch (SAXException e)
        {
            log.error("Could not parse config XML", e);
            throw new ConfigurationException(e.getMessage());
        }
        catch (IOException e)
        {
            log.error("Could not read config from stream", e);
            throw new ConfigurationException(e.getMessage());
        }
        catch (ParserConfigurationException e)
        {
            log.fatal("Could not obtain SAX parser", e);
            throw new ConfigurationException(e.getMessage());
        }
        catch (RuntimeException e)
        {
            log.fatal("RuntimeException", e);
            throw e;
        }
        catch (Throwable e)
        {
            log.fatal("Exception", e);
            throw new ConfigurationException(e.getMessage());
        }
    }

    public DefaultHandler getOSUserConfigurationHandler()
    {
        if (configHandler == null)
            configHandler = new DefaultOSUConfigurationHandler(this);

        return configHandler;
    }

    public void setOSUserConfigurationHandler(DefaultOSUConfigurationHandler configHandler)
    {
        this.configHandler = configHandler;
    }

    public void addProvider(String providerClassName, Properties providerProperties) throws ConfigurationException
    {
        if (log.isDebugEnabled())
        {
            log.debug("UserProvider class = " + providerClassName + " " + providerProperties);
        }

        try
        {
            UserProvider provider;
            provider = (UserProvider) ClassLoaderUtil.loadClass(providerClassName, this.getClass()).newInstance();

            if (provider == null)
            {
                throw new ConfigurationException("OSUser provider class was unable to be instantiated: [" + providerClassName + "]");
            }

            // set a caching manager on providers that need it
            if (provider instanceof CacheFactoryAware)
                ((CacheFactoryAware) provider).setCacheFactory(cacheFactory);

            if (provider.init(providerProperties))
            {
                if (provider instanceof AccessProvider)
                {
                    if (accessor != null)
                        accessor.setAccessProvider((AccessProvider) provider);

                    accessProvider = (AccessProvider) provider;
                }

                if (provider instanceof CredentialsProvider)
                {
                    CredentialsProvider credentialsProvider = (CredentialsProvider) provider;
                    if (accessor != null)
                        accessor.setCredentialsProvider(credentialsProvider);

                    // if the XML is parsed more than once, we only want to add each provider once
                    if (!credentialProviders.contains(credentialsProvider))
                        credentialProviders.add(credentialsProvider);

                    this.credentialsProvider = (CredentialsProvider) provider;
                }

                if (provider instanceof ProfileProvider)
                {
                    if (accessor != null)
                        accessor.setProfileProvider((ProfileProvider) provider);
                    profileProvider = (ProfileProvider) provider;
                }
            }
            else
            {
                log.error("Could not getConnectionPoolProperties provider " + providerClassName);
                throw new ConfigurationException("Could not getConnectionPoolProperties provider " + providerClassName);
            }
        }
        catch (Exception e)
        {
            log.error("Could not create instance of provider [" + providerClassName + "]: ", e);
            throw new ConfigurationException(e.getMessage());
        }
    }

    public void setAccessor(DefaultOSUAccessor accessor)
    {
        this.accessor = accessor;
    }

    public OSUAccessor getOSUAccessor()
    {
        return accessor;
    }

    public List getCredentialProviders()
    {
        return credentialProviders;
    }

    public ProfileProvider getProfileProvider()
    {
        return profileProvider;
    }

    public AccessProvider getAccessProvider()
    {
        return accessProvider;
    }

    public CredentialsProvider getCredentialsProvider()
    {
        return credentialsProvider;
    }

    public void setCacheFactory(CacheFactory cacheFactory)
    {
        this.cacheFactory = cacheFactory;
    }
}
