package com.atlassian.user.impl.ldap.search.query;

/**
 * Note - any LDAP search query returned from this parserEntity must also be related, finally, to an appropriate
 * base context in the LDAP DIT. This could be a context under which all group and user entries or stored
 * or separate contexts for each entity type. The global context will be specified in the JNDI context which
 * binds to the LDAP server.
 */

import com.atlassian.user.EntityException;
import com.atlassian.user.Group;
import com.atlassian.user.User;
import com.atlassian.user.impl.ldap.DefaultLDAPGroupFactory;
import com.atlassian.user.impl.ldap.LDAPGroupFactory;
import com.atlassian.user.impl.ldap.LDAPUserFactory;
import com.atlassian.user.impl.ldap.LiteralFilter;
import com.atlassian.user.impl.ldap.adaptor.LDAPGroupAdaptor;
import com.atlassian.user.impl.ldap.properties.LdapMembershipProperties;
import com.atlassian.user.impl.ldap.properties.LdapSearchProperties;
import com.atlassian.user.impl.ldap.repository.LdapContextFactory;
import com.atlassian.user.impl.ldap.search.DefaultLDAPUserAdaptor;
import com.atlassian.user.impl.ldap.search.LDAPPagerInfo;
import com.atlassian.user.impl.ldap.search.LDAPUserAdaptor;
import com.atlassian.user.impl.ldap.search.LdapFilterFactory;
import com.atlassian.user.impl.ldap.search.page.LDAPEntityPager;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.DefaultSearchResult;
import com.atlassian.user.search.SearchResult;
import com.atlassian.user.search.page.Pager;
import com.atlassian.user.search.query.*;
import com.atlassian.util.profiling.UtilTimerStack;
import net.sf.ldaptemplate.support.LdapEncoder;

import java.util.Iterator;

public class LDAPEntityQueryParser implements EntityQueryParser
{
    public static final String OPEN_PARAN = "(";
    public static final String CLOSE_PARAN = ")";
    public static final String EQ = "=";
    public static final String AND = "&";
    public static final String OR = "|";
    public static final String WILDCARD = TermQuery.WILDCARD;

    private final LdapContextFactory repository;
    private final LDAPUserAdaptor userAdaptor;
    private final LDAPGroupAdaptor groupAdaptor;
    private final LDAPGroupFactory groupFactory;
    private final LDAPUserFactory userFactory;

    private final RepositoryIdentifier repositoryIdentifier;
    private final LdapSearchProperties searchProperties;
    private final LdapMembershipProperties membershipProperties;

    public LDAPEntityQueryParser(LdapContextFactory repository, LDAPGroupAdaptor groupAdaptor,
        RepositoryIdentifier repositoryIdentifier, LDAPUserFactory userFactory,
        LdapSearchProperties searchProperties, LdapMembershipProperties membershipProperties,
        LdapFilterFactory filterFactory)
    {
        this.repositoryIdentifier = repositoryIdentifier;
        this.repository = repository;
        this.groupAdaptor = groupAdaptor;
        this.userFactory = userFactory;
        this.userAdaptor = new DefaultLDAPUserAdaptor(this.repository, searchProperties, filterFactory);
        this.groupFactory = new DefaultLDAPGroupFactory(searchProperties, membershipProperties);
        this.searchProperties = searchProperties;
        this.membershipProperties = membershipProperties;
    }

    public SearchResult<User> findUsers(Query<User> query) throws EntityException
    {
        if (UtilTimerStack.isActive())
            UtilTimerStack.push(this.getClass().getName() + "_findUsers");

        String parsedQuery = null;
        Pager<User> iter;
        parsedQuery = directQuery(query, parsedQuery);

        LDAPPagerInfo info = userAdaptor.search(new LiteralFilter(parsedQuery));
        iter = new LDAPEntityPager<User>(searchProperties, repository, userFactory, info);
        DefaultSearchResult<User> searchResult = new DefaultSearchResult<User>(iter, repositoryIdentifier.getKey());

        if (UtilTimerStack.isActive())
            UtilTimerStack.pop(this.getClass().getName() + "_findUsers");

        return searchResult;
    }

    public SearchResult<Group> findGroups(Query<Group> query) throws EntityException
    {
        String parsedQuery = directQuery(query, null);
        LDAPPagerInfo info = groupAdaptor.search(new LiteralFilter(parsedQuery));
        Pager<Group> pager = new LDAPEntityPager<Group>(searchProperties, repository, groupFactory, info);
        return new DefaultSearchResult<Group>(pager, repositoryIdentifier.getKey());
    }

    public SearchResult<User> findUsers(Query<User> query, QueryContext context) throws EntityException
    {
        if (!context.contains(repositoryIdentifier))
            return null;

        return findUsers(query);
    }

    public SearchResult<Group> findGroups(Query<Group> query, QueryContext context) throws EntityException
    {
        if (!context.contains(repositoryIdentifier))
            return null;

        return findGroups(query);
    }

    private String directQuery(Query query, String defaultQuery) throws EntityException
    {
        if (query instanceof TermQuery)
        {
            StringBuffer parsedQueryStringBuffer = parseQuery((TermQuery) query);
            return parsedQueryStringBuffer.toString();
        }
        else if (query instanceof BooleanQuery)
        {
            return parseQuery((BooleanQuery) query).toString();
        }

        return defaultQuery;
    }

    public StringBuffer parseQuery(BooleanQuery query) throws EntityException
    {
        StringBuffer parsedClause = new StringBuffer();
        parsedClause.append(OPEN_PARAN);

        if (query.isAND())
            parsedClause.append(AND);
        else
            parsedClause.append(OR);

        Iterator queryIter = query.getQueries().iterator();

        while (queryIter.hasNext())
        {
            Query foundQuery = (Query) queryIter.next();
            if (foundQuery instanceof BooleanQuery)
                parsedClause.append(parseQuery((BooleanQuery) foundQuery));
            else
                parsedClause.append(parseQuery((TermQuery) foundQuery));
        }

        parsedClause.append(CLOSE_PARAN);

        return parsedClause;
    }

    public StringBuffer parseQuery(TermQuery q) throws EntityException
    {
        StringBuffer parsedQuery = null;

        if (q instanceof UserNameTermQuery)
            parsedQuery = parseTermQuery(q, searchProperties.getUsernameAttribute());
        else if (q instanceof GroupNameTermQuery)
            parsedQuery = parseTermQuery(q, searchProperties.getGroupnameAttribute());
        else if (q instanceof EmailTermQuery)
            parsedQuery = parseTermQuery(q, searchProperties.getEmailAttribute());
        else if (q instanceof FullNameTermQuery)
            parsedQuery = parseFullNameTermQuery(q);

        return parsedQuery;
    }

    private StringBuffer parseFullNameTermQuery(TermQuery q)
    {
        StringBuffer query = new StringBuffer();

        query.insert(0, OR);
        query.insert(0, "(");
        query.append(parseTermQuery(q, searchProperties.getFirstnameAttribute()));
        query.append(parseTermQuery(q, searchProperties.getSurnameAttribute()));
        query.append(")");

        return query;
    }

    /**
     * Generic method which can parse a term query and relate it to a targeted attributeType in LDAP speak.
     * <p/>
     * If attributeType is 'cn' and q.getTerm() is 'fred' then it will return a {@link StringBuffer} holding (cn=fred).
     * It will also deal with substring matching rules. If the {@link TermQuery} was matching substrings using {@link
     * TermQuery#SUBSTRING_STARTS_WITH}, using the example values above, it would return (cn=fred*)
     */
    private StringBuffer parseTermQuery(TermQuery q, String attributeType)
    {
        StringBuffer parsedQuery;
        parsedQuery = new StringBuffer();
        parsedQuery.append(OPEN_PARAN);
        parsedQuery.append(attributeType);
        parsedQuery.append(EQ);

        if (q.isMatchingSubstring())
        {
            if ((q.getMatchingRule().equals(TermQuery.SUBSTRING_ENDS_WITH)) ||
                (q.getMatchingRule().equals(TermQuery.SUBSTRING_CONTAINS)))
            {
                parsedQuery.append(WILDCARD);
            }
        }

        parsedQuery.append(LdapEncoder.filterEncode(q.getTerm()));

        if (q.isMatchingSubstring())
        {
            if ((q.getMatchingRule().equals(TermQuery.SUBSTRING_STARTS_WITH)) ||
                (q.getMatchingRule().equals(TermQuery.SUBSTRING_CONTAINS)))
            {
                parsedQuery.append(WILDCARD);
            }
        }

        parsedQuery.append(CLOSE_PARAN);
        return parsedQuery;
    }
}
