package com.atlassian.user.impl.ldap;

import com.atlassian.user.Entity;
import com.atlassian.user.EntityException;
import com.atlassian.user.User;
import com.atlassian.user.impl.ReadOnlyUserManager;
import com.atlassian.user.impl.RepositoryException;
import com.atlassian.user.impl.ldap.properties.LdapSearchProperties;
import com.atlassian.user.impl.ldap.repository.LdapContextFactory;
import com.atlassian.user.impl.ldap.search.DefaultLDAPUserAdaptor;
import com.atlassian.user.impl.ldap.search.LDAPPagerInfo;
import com.atlassian.user.impl.ldap.search.LDAPUserAdaptor;
import com.atlassian.user.impl.ldap.search.LdapFilterFactory;
import com.atlassian.user.impl.ldap.search.page.LDAPEntityPager;
import com.atlassian.user.impl.ldap.search.page.LDAPSingleStringPager;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.page.Pager;
import com.atlassian.util.profiling.UtilTimerStack;
import net.sf.ldaptemplate.support.filter.EqualsFilter;
import org.apache.log4j.Logger;

public class LDAPUserManagerReadOnly extends ReadOnlyUserManager
{
    protected final Logger log = Logger.getLogger(this.getClass());

    private final RepositoryIdentifier repositoryIdentifier;
    private final LdapContextFactory repository;
    private final LdapSearchProperties searchProperties;
    private final LDAPUserAdaptor userAdaptor;

    public LDAPUserManagerReadOnly(RepositoryIdentifier repositoryIdentifier, LdapContextFactory repository,
        LdapSearchProperties searchProperties, LdapFilterFactory filterFactory)
    {
        this.repositoryIdentifier = repositoryIdentifier;
        this.repository = repository;
        this.searchProperties = searchProperties;
        userAdaptor = new DefaultLDAPUserAdaptor(repository, searchProperties, filterFactory);
    }

    /**
     * @return all users in the configured repository
     * @throws RepositoryException
     */
    public Pager<User> getUsers() throws EntityException
    {
        profilePush(this.getClass().getName() + "_getUsers");

        LDAPPagerInfo info = userAdaptor.search(null);
        Pager<User> pager = new LDAPEntityPager<User>(searchProperties, repository, new DefaultLDAPUserFactory(searchProperties), info);

        profilePop(this.getClass().getName() + "_getUsers");

        return pager;
    }

    public Pager<String> getUserNames() throws EntityException
    {
        LDAPPagerInfo info = userAdaptor.search(null, new String[]{searchProperties.getUsernameAttribute()});
        return new LDAPSingleStringPager(searchProperties, repository, info);
    }

    public User getUser(String username) throws EntityException
    {
        profilePush(this.getClass().getName() + "_getUser(" + username + ")");

        User user = null;

        try
        {
            LDAPPagerInfo info =
                userAdaptor.search(new EqualsFilter(searchProperties.getUsernameAttribute(), username));

            Pager pager = new LDAPEntityPager<User>(searchProperties, repository, new DefaultLDAPUserFactory(searchProperties), info);

            if (pager.getCurrentPage().size() > 0)
                user = (User) pager.getCurrentPage().get(0);
        }
        catch (EntityException e)
        {
            log.error("Error retrieving user: '" + username + "' from LDAP.", e);
        }

        profilePop(this.getClass().getName() + "_getUser(" + username + ")");

        return user;
    }

    /**
     * @return the {@link com.atlassian.user.repository.RepositoryIdentifier} which is managed by this instance.
     */
    public RepositoryIdentifier getIdentifier()
    {
        return repositoryIdentifier;
    }

    public RepositoryIdentifier getRepository(Entity entity) throws EntityException
    {
        if (!LDAPValidator.validateLDAPEntity(entity))
            return null;

        if (getUser(entity.getName()) == null) // not handled by this manager
            return null;

        return repositoryIdentifier;

    }

    private void profilePush(String key)
    {
        if (UtilTimerStack.isActive())
            UtilTimerStack.push(key);
    }

    private void profilePop(String key)
    {
        if (UtilTimerStack.isActive())
            UtilTimerStack.pop(key);
    }
}