package com.atlassian.user.impl.delegation.security.authentication;

import com.atlassian.user.EntityException;
import com.atlassian.user.UserManager;
import com.atlassian.user.impl.delegation.repository.DelegatingRepository;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.security.authentication.Authenticator;
import com.atlassian.util.profiling.UtilTimerStack;
import org.apache.log4j.Logger;

import java.util.List;
import java.util.LinkedList;
import java.util.ArrayList;

public class DelegatingAuthenticator implements Authenticator
{
    private static final Logger log = Logger.getLogger(DelegatingAuthenticator.class);
    private final List<Authenticator> authenticators;
    private final UserManager userManager;

    public DelegatingAuthenticator(UserManager userManager, List<Authenticator> authenticators)
    {
        this.userManager = userManager;
        this.authenticators = authenticators;
    }

    public boolean authenticate(String username, String password) throws EntityException
    {
        if (UtilTimerStack.isActive())
            UtilTimerStack.push(this.getClass().getName() + "_delegating_authenticate(" + username + ")");

        try
        {
            RepositoryIdentifier repository = userManager.getRepository(userManager.getUser(username));
            List<Authenticator> authenticators = getAuthenticatorsForRepository(repository);

            if (authenticators.isEmpty())
            {
                log.error("Failed to find authenticator for user " + username + " from repository " + repository);
                return false;
            }

            for (Authenticator authenticator : authenticators)
            {
                if (authenticate(authenticator, username, password))
                    return true;
            }
        }
        finally
        {
            if (UtilTimerStack.isActive())
                UtilTimerStack.pop(this.getClass().getName() + "_delegating_authenticate(" + username + ")");
        }

        return false;
    }

    private boolean authenticate(Authenticator authenticator, String username, String password)
    {
        try
        {
            return authenticator.authenticate(username, password);
        }
        catch (EntityException e)
        {
            log.error(authenticator.getRepository() + ": " + e.getMessage());
            return false;
        }
    }

    private List<Authenticator> getAuthenticatorsForRepository(RepositoryIdentifier repository)
    {
        List<Authenticator> result = new LinkedList<Authenticator>();

        for (Authenticator authenticator : authenticators)
        {
            if (authenticator.getRepository().equals(repository))
                result.add(authenticator);
        }
        return result;
    }

    public RepositoryIdentifier getRepository()
    {
        List<RepositoryIdentifier> repositories = new ArrayList<RepositoryIdentifier>(authenticators.size());

        for (Authenticator authenticator : authenticators)
        {
            repositories.add(authenticator.getRepository());
        }

        return new DelegatingRepository(repositories);
    }

    public List getAuthenticators()
    {
        return authenticators;
    }

}
