package com.atlassian.user.impl.delegation.search.query;

import com.atlassian.user.EntityException;
import com.atlassian.user.search.DefaultSearchResult;
import com.atlassian.user.search.SearchResult;
import com.atlassian.user.search.query.AllRepositoriesQueryContext;
import com.atlassian.user.search.query.EntityQueryParser;
import com.atlassian.user.search.query.Query;
import com.atlassian.user.search.query.QueryContext;
import org.apache.log4j.Logger;

import java.util.Iterator;
import java.util.List;

public class DelegatingEntityQueryParser implements EntityQueryParser
{
    private static final Logger log = Logger.getLogger(DelegatingEntityQueryParser.class);
    private final List<EntityQueryParser> entityQueryParsers;

    public DelegatingEntityQueryParser(List<EntityQueryParser> entityQueryParsers)
    {
        this.entityQueryParsers = entityQueryParsers;
    }

    /**
     * Generates a {@link com.atlassian.user.search.SearchResult} object for each {@link com.atlassian.user.repository.RepositoryIdentifier} the
     * entityQueryParser can search. These are grouped in an array and returned.
     * <p/>
     * Thus, if the entityQueryParser only knows about one configuration, the array
     * will either have a size of 0 (no sucesses) or 1.
     */
    public SearchResult findUsers(Query query) throws EntityException
    {
        return findUsers(query, new AllRepositoriesQueryContext());
    }

    /**
     * Generates a {@link com.atlassian.user.search.SearchResult} object for each {@link com.atlassian.user.repository.RepositoryIdentifier} the
     * entityQueryParser can search. These are grouped in an array and returned.
     * <p/>
     * Thus, if the entityQueryParser only knows about one configuration, the array
     * will either have a size of 0 (no sucesses) or 1.
     */
    public SearchResult findGroups(Query query) throws EntityException
    {
        return findGroups(query, new AllRepositoriesQueryContext());
    }

    public SearchResult findUsers(Query query, QueryContext context) throws EntityException
    {
        Iterator iter = entityQueryParsers.iterator();
        DefaultSearchResult result = new DefaultSearchResult();
        boolean withinContext = false;

        while (iter.hasNext())
        {
            EntityQueryParser parser = (EntityQueryParser) iter.next();

            SearchResult returned = null;

            try
            {
                returned = parser.findUsers(query, context);
            }
            catch (EntityException e)
            {
                log.info(e.getMessage());
            }

            if (returned != null)
            {
                withinContext = true;
                Iterator keysIter = returned.repositoryKeyset().iterator();

                while (keysIter.hasNext())
                {
                    String repokey = (String) keysIter.next();
                    result.addToResults(repokey, returned.pager(repokey));
                }
            }
        }

        // if the search was not within the context, return null as per the method contract
        return (withinContext) ? result : null;
    }

    public SearchResult findGroups(Query query, QueryContext context) throws EntityException
    {
        Iterator iter = entityQueryParsers.iterator();
        DefaultSearchResult result = new DefaultSearchResult();
        boolean withinContext = false;

        while (iter.hasNext())
        {
            EntityQueryParser parser = (EntityQueryParser) iter.next();

            SearchResult returned = null;

            try
            {
                returned = parser.findGroups(query, context);
            }
            catch (EntityException e)
            {
                log.info(e.getMessage());
            }

            if (returned != null)
            {
                withinContext = true;
                Iterator keysIter = returned.repositoryKeyset().iterator();

                while (keysIter.hasNext())
                {
                    String repokey = (String) keysIter.next();
                    result.addToResults(repokey, returned.pager(repokey));
                }
            }
        }

        // if the search was not within the context, return null as per the method contract
        return (withinContext) ? result : null;
    }
}
