package com.atlassian.user.impl.ldap.search.query;

/**
 * Note - any LDAP search query returned from this parserEntity must also be related, finally, to an appropriate
 * base context in the LDAP DIT. This could be a context under which all group and user entries or stored
 * or separate contexts for each entity type. The global context will be specified in the JNDI context which
 * binds to the LDAP server.
 */

import com.atlassian.user.EntityException;
import com.atlassian.user.User;
import com.atlassian.user.Group;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.impl.RepositoryException;
import com.atlassian.user.impl.ldap.DefaultLDAPGroupFactory;
import com.atlassian.user.impl.ldap.LDAPGroupFactory;
import com.atlassian.user.impl.ldap.LDAPUserFactory;
import com.atlassian.user.impl.ldap.LiteralFilter;
import com.atlassian.user.impl.ldap.adaptor.LDAPGroupAdaptor;
import com.atlassian.user.impl.ldap.properties.LdapSearchProperties;
import com.atlassian.user.impl.ldap.properties.LdapMembershipProperties;
import com.atlassian.user.impl.ldap.repository.LdapContextFactory;
import com.atlassian.user.impl.ldap.search.DefaultLDAPUserAdaptor;
import com.atlassian.user.impl.ldap.search.LDAPPagerInfo;
import com.atlassian.user.impl.ldap.search.LDAPUserAdaptor;
import com.atlassian.user.impl.ldap.search.LdapFilterFactory;
import com.atlassian.user.impl.ldap.search.page.LDAPEntityPager;
import com.atlassian.user.search.DefaultSearchResult;
import com.atlassian.user.search.SearchResult;
import com.atlassian.user.search.page.Pager;
import com.atlassian.user.search.query.*;
import com.atlassian.util.profiling.UtilTimerStack;
import net.sf.ldaptemplate.support.LdapEncoder;

import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import java.util.Iterator;

public class LDAPEntityQueryParser implements EntityQueryParser
{
    public static final String OPEN_PARAN = "(";
    public static final String CLOSE_PARAN = ")";
    public static final String EQ = "=";
    public static final String AND = "&";
    public static final String OR = "|";
    public static final String WILDCARD = TermQuery.WILDCARD;

    private final LdapContextFactory repository;
    private final LDAPUserAdaptor userAdaptor;
    private final LDAPGroupAdaptor groupAdaptor;
    private final LDAPGroupFactory groupFactory;
    private final LDAPUserFactory userFactory;

    private final RepositoryIdentifier repositoryIdentifier;
    private final LdapSearchProperties searchProperties;
    private final LdapMembershipProperties membershipProperties;

    public LDAPEntityQueryParser(LdapContextFactory repository, LDAPGroupAdaptor groupAdaptor,
        RepositoryIdentifier repositoryIdentifier, LDAPUserFactory userFactory,
        LdapSearchProperties searchProperties, LdapMembershipProperties membershipProperties,
        LdapFilterFactory filterFactory)
    {
        this.repositoryIdentifier = repositoryIdentifier;
        this.repository = repository;
        this.groupAdaptor = groupAdaptor;
        this.userFactory = userFactory;
        this.userAdaptor = new DefaultLDAPUserAdaptor(this.repository, searchProperties, filterFactory);
        this.groupFactory = new DefaultLDAPGroupFactory(searchProperties, membershipProperties);
        this.searchProperties = searchProperties;
        this.membershipProperties = membershipProperties;
    }

    public com.atlassian.user.search.SearchResult findUsers(Query query) throws EntityException
    {
        if (UtilTimerStack.isActive())
            UtilTimerStack.push(this.getClass().getName() + "_findUsers");

        String parsedQuery = null;
        Pager iter;
        parsedQuery = directQuery(query, parsedQuery);

        LDAPPagerInfo info = userAdaptor.search(new LiteralFilter(parsedQuery));
        iter = new LDAPEntityPager<User>(searchProperties, repository, userFactory, info);
        DefaultSearchResult searchResult = new DefaultSearchResult(iter, repositoryIdentifier.getKey());

        if (UtilTimerStack.isActive())
            UtilTimerStack.pop(this.getClass().getName() + "_findUsers");

        return searchResult;
    }

    public com.atlassian.user.search.SearchResult findGroups(Query query) throws EntityException
    {
        String parsedQuery = directQuery(query, null);
        LDAPPagerInfo info = groupAdaptor.search(new LiteralFilter(parsedQuery));
        Pager pager = new LDAPEntityPager<Group>(searchProperties, repository, groupFactory, info);
        return new DefaultSearchResult(pager, repositoryIdentifier.getKey());
    }

    public SearchResult findUsers(Query query, QueryContext context) throws EntityException
    {
        if (!context.contains(repositoryIdentifier))
            return null;

        return findUsers(query);
    }

    public SearchResult findGroups(Query query, QueryContext context) throws EntityException
    {
        if (!context.contains(repositoryIdentifier))
            return null;

        return findGroups(query);
    }

    private String directQuery(Query query, String defaultQuery) throws EntityException
    {
        if (query instanceof TermQuery)
        {
            StringBuffer parsedQueryStringBuffer = parseQuery((TermQuery) query);
            return parsedQueryStringBuffer.toString();
        }
        else if (query instanceof BooleanQuery)
        {
            return parseQuery((BooleanQuery) query).toString();
        }

        return defaultQuery;
    }

    public StringBuffer parseQuery(BooleanQuery query) throws EntityException
    {
        StringBuffer parsedClause = new StringBuffer();
        parsedClause.append(OPEN_PARAN);

        if (query.isAND())
            parsedClause.append(AND);
        else
            parsedClause.append(OR);

        Iterator queryIter = query.getQueries().iterator();

        while (queryIter.hasNext())
        {
            Query foundQuery = (Query) queryIter.next();
            if (foundQuery instanceof BooleanQuery)
                parsedClause.append(parseQuery((BooleanQuery) foundQuery));
            else
                parsedClause.append(parseQuery((TermQuery) foundQuery));
        }

        parsedClause.append(CLOSE_PARAN);

        return parsedClause;
    }

    public StringBuffer parseQuery(TermQuery q) throws EntityException
    {
        StringBuffer parsedQuery = null;

        if (q instanceof UserNameTermQuery)
            parsedQuery = parseTermQuery(q, searchProperties.getUsernameAttribute());
        else if (q instanceof GroupNameTermQuery)
            parsedQuery = parseTermQuery(q, searchProperties.getGroupnameAttribute());
        else if (q instanceof EmailTermQuery)
            parsedQuery = parseTermQuery(q, searchProperties.getEmailAttribute());
        else if (q instanceof FullNameTermQuery)
            parsedQuery = parseFullNameTermQuery(q);
        else if (q instanceof UsersInGroupTwoTermQuery)
            parsedQuery = parseMemberNamesInGroupTermQuery();
        else if (q instanceof GroupsOfUserTwoTermQuery)
            parsedQuery = parseGroupsOfUserTwoTermQuery((GroupsOfUserTwoTermQuery) q);

        return parsedQuery;
    }

    private StringBuffer parseFullNameTermQuery(TermQuery q)
    {
        StringBuffer query = new StringBuffer();

        query.insert(0, OR);
        query.insert(0, "(");
        query.append(parseTermQuery(q, searchProperties.getFirstnameAttribute()));
        query.append(parseTermQuery(q, searchProperties.getSurnameAttribute()));
        query.append(")");

        return query;
    }

    /**
     * Looks up membership lists associated with the group indicated by {@link UsersInGroupTwoTermQuery} and returns an
     * ldap getGroupEntries string capable of returning their dns.
     */
    private StringBuffer parseMemberNamesInGroupTermQuery() throws EntityException
    {
        if (!membershipProperties.isMembershipAttributeOnGroup())
            throw new UnsupportedOperationException();

        LDAPPagerInfo pagerInfo = groupAdaptor.getGroupEntries(null);
        NamingEnumeration enume = pagerInfo.getNamingEnumeration();
        return parseMembershipAttributesForGroupNames(enume);
    }

    /**
     * @return StringBuffer holding an LDAP getGroupEntries query which will return a list of the user's groups.
     */
    private StringBuffer parseGroupsOfUserTwoTermQuery(GroupsOfUserTwoTermQuery q) throws EntityException
    {
        String userPattern = null;

        UserNameTermQuery uQuery = q.getUserNameTermQuery();
        if (uQuery.isMatchingSubstring())
        {
            if (uQuery.getMatchingRule().equals(TermQuery.SUBSTRING_STARTS_WITH))
                userPattern = uQuery.getTerm() + WILDCARD;
            else if (uQuery.getMatchingRule().equals(TermQuery.SUBSTRING_ENDS_WITH))
                userPattern = WILDCARD + uQuery.getTerm();
            else if (uQuery.getMatchingRule().equals(TermQuery.SUBSTRING_CONTAINS))
                userPattern = WILDCARD + uQuery.getTerm() + WILDCARD;
        }
        else
            userPattern = uQuery.getTerm();

        //First we find the membership information for the user via the groupAdaptor
        LDAPPagerInfo pagerInfo = groupAdaptor.getGroupEntriesViaMembership(userPattern);

        StringBuffer groupQuery;

        if (membershipProperties.isMembershipAttributeOnGroup())
            groupQuery = buildQueryForUsersInStaticGroups(pagerInfo.getNamingEnumeration());
        else
            groupQuery = buildQueryForUsersInDynamicGroups(pagerInfo.getNamingEnumeration());

        return groupQuery;
    }

    private StringBuffer buildQueryForUsersInDynamicGroups(NamingEnumeration enume) throws EntityException
    {
        StringBuffer parsedQuery = null;

        while (enume.hasMoreElements())
        {
            javax.naming.directory.SearchResult result = (javax.naming.directory.SearchResult) enume.nextElement();
            Attributes attrs = result.getAttributes();

            Attribute attr = attrs.get(membershipProperties.getMembershipAttribute());

            if (attr == null)
                return null;

            try
            {
                NamingEnumeration membershipEnume = attr.getAll();
                while (membershipEnume.hasMoreElements())
                {
                    boolean wrapping = false;

                    if (parsedQuery == null)
                    {
                        parsedQuery = new StringBuffer();
                    }
                    else
                    {
                        parsedQuery.insert(0, OR);
                        parsedQuery.insert(0, OPEN_PARAN);
                        wrapping = true;
                    }

                    parsedQuery.append(OPEN_PARAN);
                    parsedQuery.append(((String) membershipEnume.nextElement()).split(",")[0]);
                    parsedQuery.append(CLOSE_PARAN);

                    if (wrapping)
                        parsedQuery.append(CLOSE_PARAN);
                }
            }
            catch (Exception e)
            {
                throw new RepositoryException(e);
            }
        }

        return parsedQuery;
    }

    private StringBuffer buildQueryForUsersInStaticGroups(NamingEnumeration enume) throws EntityException
    {
        StringBuffer parsedQuery = null;

        while (enume.hasMoreElements())
        {
            javax.naming.directory.SearchResult result = (javax.naming.directory.SearchResult) enume.nextElement();
            Attributes attrs = result.getAttributes();

            Attribute attr = attrs.get(searchProperties.getGroupnameAttribute());

            if (attr == null)
                return null;

            try
            {
                NamingEnumeration membershipEnume = attr.getAll();
                while (membershipEnume.hasMoreElements())
                {
                    boolean wrapping = false;

                    if (parsedQuery == null)
                    {
                        parsedQuery = new StringBuffer();
                    }
                    else
                    {
                        parsedQuery.insert(0, OR);
                        parsedQuery.insert(0, OPEN_PARAN);
                        wrapping = true;
                    }

                    parsedQuery.append(OPEN_PARAN);
                    parsedQuery.append(searchProperties.getGroupnameAttribute());
                    parsedQuery.append(EQ);
                    parsedQuery.append(((String) membershipEnume.nextElement()).split(",")[0]);
                    parsedQuery.append(CLOSE_PARAN);

                    if (wrapping)
                        parsedQuery.append(CLOSE_PARAN);
                }
            }
            catch (Exception e)
            {
                throw new RepositoryException(e);
            }
        }

        return parsedQuery;
    }

    private StringBuffer parseMembershipAttributesForGroupNames(NamingEnumeration enume)
    {
        StringBuffer parsedQuery = null;

        while (enume.hasMoreElements())
        {
            javax.naming.directory.SearchResult result = (javax.naming.directory.SearchResult) enume.nextElement();
            Attributes attrs = result.getAttributes();

            Attribute attr;

            if (membershipProperties.isMembershipAttributeOnGroup())
                attr = attrs.get(membershipProperties.getMembershipAttribute());
            else
                attr = attrs.get(searchProperties.getGroupnameAttribute());

            try
            {
                if (parsedQuery == null)
                    parsedQuery = parseMembershipAttributeForGroupNames(attr);
                else
                {
                    parsedQuery.insert(0, "(|");
                    parsedQuery = parsedQuery.append(parseMembershipAttributeForGroupNames(attr));
                    parsedQuery.append(")");
                }
            }
            catch (Exception e)
            {
                e.printStackTrace();
            }
        }
        return parsedQuery;
    }

    private StringBuffer parseMembershipAttributeForGroupNames(Attribute attr) throws NamingException
    {
        if (attr == null)
            return null;

        StringBuffer parsedQuery = null;
        NamingEnumeration membershipEnume = attr.getAll();
        while (membershipEnume.hasMoreElements())
        {
            boolean wrapping = false;

            if (parsedQuery == null)
            {
                parsedQuery = new StringBuffer();
            }
            else
            {
                parsedQuery.insert(0, OR);
                parsedQuery.insert(0, OPEN_PARAN);
                wrapping = true;
            }

            if (membershipProperties.isMembershipAttributeOnGroup())
            {
                parsedQuery.append(OPEN_PARAN);
                parsedQuery.append(((String) membershipEnume.nextElement()).split(",")[0]);
                parsedQuery.append(CLOSE_PARAN);
            }
            else
            {
                parsedQuery.append(OPEN_PARAN);
                parsedQuery.append(searchProperties.getGroupnameAttribute());
                parsedQuery.append(EQ);
                parsedQuery.append(((String) membershipEnume.nextElement()).split(",")[0]);
                parsedQuery.append(CLOSE_PARAN);

            }

            if (wrapping)
                parsedQuery.append(CLOSE_PARAN);
        }

        return parsedQuery;
    }

    /**
     * Generic method which can parse a term query and relate it to a targeted attributeType in LDAP speak.
     * <p/>
     * If attributeType is 'cn' and q.getTerm() is 'fred' then it will return a {@link StringBuffer} holding (cn=fred).
     * It will also deal with substring matching rules. If the {@link TermQuery} was matching substrings using {@link
     * TermQuery#SUBSTRING_STARTS_WITH}, using the example values above, it would return (cn=fred*)
     */
    private StringBuffer parseTermQuery(TermQuery q, String attributeType)
    {
        StringBuffer parsedQuery;
        parsedQuery = new StringBuffer();
        parsedQuery.append(OPEN_PARAN);
        parsedQuery.append(attributeType);
        parsedQuery.append(EQ);

        if (q.isMatchingSubstring())
        {
            if ((q.getMatchingRule().equals(TermQuery.SUBSTRING_ENDS_WITH)) ||
                (q.getMatchingRule().equals(TermQuery.SUBSTRING_CONTAINS)))
            {
                parsedQuery.append(WILDCARD);
            }
        }

        parsedQuery.append(LdapEncoder.filterEncode(q.getTerm()));

        if (q.isMatchingSubstring())
        {
            if ((q.getMatchingRule().equals(TermQuery.SUBSTRING_STARTS_WITH)) ||
                (q.getMatchingRule().equals(TermQuery.SUBSTRING_CONTAINS)))
            {
                parsedQuery.append(WILDCARD);
            }
        }

        parsedQuery.append(CLOSE_PARAN);
        return parsedQuery;
    }
}
