package com.atlassian.user.impl.ldap.adaptor;

import com.atlassian.user.EntityException;
import com.atlassian.user.Group;
import com.atlassian.user.User;
import com.atlassian.user.impl.RepositoryException;
import com.atlassian.user.impl.ldap.LDAPGroupFactory;
import com.atlassian.user.impl.ldap.LDAPUserFactory;
import com.atlassian.user.impl.ldap.properties.LdapSearchProperties;
import com.atlassian.user.impl.ldap.properties.LdapMembershipProperties;
import com.atlassian.user.impl.ldap.repository.LdapContextFactory;
import com.atlassian.user.impl.ldap.search.DefaultLDAPUserAdaptor;
import com.atlassian.user.impl.ldap.search.LDAPPagerInfo;
import com.atlassian.user.impl.ldap.search.LDAPUserAdaptor;
import com.atlassian.user.impl.ldap.search.LdapFilterFactory;
import com.atlassian.user.impl.ldap.search.page.LDAPListOfGroupsPager;
import com.atlassian.user.impl.ldap.search.page.LDAPSingleStringPager;
import com.atlassian.user.search.page.Pager;
import net.sf.ldaptemplate.support.filter.EqualsFilter;
import net.sf.ldaptemplate.support.filter.Filter;

import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.SearchResult;

/**
 * Dynamic groups, in 'LDAP speak', express their membership via user entries (you need
 * to examine the user entry to discover what groups the user belongs to).
 *
 * The role of this class is locate dynamic groups via a getGroupEntries and call the appropriate factory
 * to construct Groups from the LDAP getGroupEntries results.
 */
public class LDAPDynamicGroupAdaptor extends AbstractLDAPGroupAdaptor
{
    private final LdapMembershipProperties membershipProperties;
    private final LDAPUserAdaptor userAdaptor;

    /** @deprecated userFactory no longer required */
    @SuppressWarnings({"UnusedDeclaration"})
    public LDAPDynamicGroupAdaptor(LdapContextFactory repo, LdapSearchProperties searchProperties,
        LDAPGroupFactory groupFactory, LdapFilterFactory filterFactory, LDAPUserFactory userFactory,
        LdapMembershipProperties membershipProperties)
    {
        super(repo, searchProperties, groupFactory, filterFactory);
        this.membershipProperties = membershipProperties;
        this.userAdaptor = new DefaultLDAPUserAdaptor(repository, searchProperties, filterFactory);
    }

    public LDAPDynamicGroupAdaptor(LdapContextFactory repo, LdapSearchProperties searchProperties,
        LDAPGroupFactory groupFactory, LdapFilterFactory filterFactory,
        LdapMembershipProperties membershipProperties)
    {
        super(repo, searchProperties, groupFactory, filterFactory);
        this.membershipProperties = membershipProperties;
        this.userAdaptor = new DefaultLDAPUserAdaptor(repository, searchProperties, filterFactory);
    }

    public Pager<Group> getGroups(User user) throws EntityException
    {
        LDAPPagerInfo answer =
            getGroupEntriesViaMembership(user.getName(), new String[]{membershipProperties.getMembershipAttribute()});
        return new LDAPListOfGroupsPager(searchProperties, repository, groupFactory, answer);
    }

    private String getGroupsForUserSearchString(NamingEnumeration enume) throws RepositoryException
    {
        String query = null;

        while (enume.hasMoreElements())
        {
            SearchResult result = (SearchResult) enume.nextElement();
            Attributes attrs = result.getAttributes();
            Attribute membershipAttribute = attrs.get(membershipProperties.getMembershipAttribute());

            if (membershipAttribute != null)
            {
                try
                {
                    NamingEnumeration groupList = membershipAttribute.getAll();

                    while (groupList.hasMoreElements())
                    {
                        String groupDN = (String) groupList.nextElement();

                        if (query == null)
                            query = "(" + groupDN.split(",")[0] + ")";
                        else
                            query = "(|" + query + "(" + groupDN.split(",")[0] + "))";
                    }
                }
                catch (NamingException e)
                {
                    throw new RepositoryException(e);
                }
            }
        }

        return query;
    }

    // TODO: Prior to genericising, findMembers on the StaticGroupAdaptor returned a list of Strings, while
    //       the dynamic group adapter method returned a list of Group entities. Since the static adaptor is
    //       the one most likely to be being used, I've reverted to that behaviour.
    public Pager<String> findMembers(Group group) throws EntityException
    {
        return findMemberNames(group);
//        LDAPPagerInfo ldapPagerInfo = findMembershipEntries(group, null);
//        return new LDAPEntityPager(repository, userFactory, ldapPagerInfo);
    }

    private LDAPPagerInfo findMembershipEntries(Group group, String[] attributesToReturn) throws EntityException
    {
        Filter searchFilter = new EqualsFilter(membershipProperties.getMembershipAttribute(), getGroupDN(group));

        LDAPPagerInfo ldapPagerInfo = null;

        try
        {
            if (attributesToReturn == null)
                ldapPagerInfo = userAdaptor.search(searchFilter);
            else
                ldapPagerInfo = userAdaptor.search(searchFilter, attributesToReturn);
        }
        catch (EntityException e)
        {
            log.fatal("Could not find users in group [" + group.getName() + "] ", e);
        }
        return ldapPagerInfo;
    }

    public Pager<String> findMemberNames(Group group) throws EntityException
    {
        LDAPPagerInfo ldapPagerInfo =
            findMembershipEntries(group, new String[]{searchProperties.getUsernameAttribute()});
        return new LDAPSingleStringPager(searchProperties, repository, ldapPagerInfo);
    }

    public boolean hasStaticGroups()
    {
        return false;
    }

    public boolean hasMembership(Group group, User user) throws EntityException
    {
        LDAPPagerInfo wrapper =
            userAdaptor.getUserAttributes(user.getName(), new String[]{membershipProperties.getMembershipAttribute()});
        NamingEnumeration enume = wrapper.getNamingEnumeration();

        return getGroupsForUserSearchString(enume).indexOf(group.getName()) != -1;
    }

    public LDAPPagerInfo getGroupEntriesViaMembership(User user) throws EntityException
    {
        return getGroupEntriesViaMembership(user.getName());
    }

    public LDAPPagerInfo getGroupEntriesViaMembership(String username) throws EntityException
    {
        return getGroupEntriesViaMembership(username, null);
    }

    public LDAPPagerInfo getGroupEntriesViaMembership(String username, String[] attributesToReturn)
        throws EntityException
    {
        if (attributesToReturn == null)
            attributesToReturn =
                new String[]{searchProperties.getGroupnameAttribute(), membershipProperties.getMembershipAttribute()};

        return userAdaptor.getUserAttributes(username, attributesToReturn);
    }
}
