package com.atlassian.user.impl.ldap.adaptor;

import com.atlassian.user.EntityException;
import com.atlassian.user.Group;
import com.atlassian.user.impl.EntityMissingException;
import com.atlassian.user.impl.RepositoryException;
import com.atlassian.user.impl.ldap.LDAPEntity;
import com.atlassian.user.impl.ldap.LDAPGroupFactory;
import com.atlassian.user.impl.ldap.properties.LdapSearchProperties;
import com.atlassian.user.impl.ldap.repository.LdapContextFactory;
import com.atlassian.user.impl.ldap.search.LDAPPagerInfo;
import com.atlassian.user.impl.ldap.search.LdapFilterFactory;
import com.atlassian.user.impl.ldap.search.page.LDAPEntityPager;
import com.atlassian.user.search.page.Pager;
import com.atlassian.user.util.LDAPUtils;
import com.atlassian.util.profiling.UtilTimerStack;
import com.opensymphony.util.TextUtils;
import net.sf.ldaptemplate.support.filter.EqualsFilter;
import net.sf.ldaptemplate.support.filter.Filter;
import org.apache.log4j.Logger;

import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attributes;
import javax.naming.directory.DirContext;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import java.text.MessageFormat;

public abstract class AbstractLDAPGroupAdaptor implements LDAPGroupAdaptor
{
    protected final Logger log = Logger.getLogger(this.getClass());

    protected final LdapSearchProperties searchProperties;
    private final LdapFilterFactory filterFactory;

    protected final LdapContextFactory repository;
    protected final LDAPGroupFactory groupFactory;

    protected AbstractLDAPGroupAdaptor(LdapContextFactory repo, LdapSearchProperties searchProperties,
        LDAPGroupFactory groupFactory, LdapFilterFactory filterFactory)
    {
        this.repository = repo;
        this.searchProperties = searchProperties;
        this.filterFactory = filterFactory;
        this.groupFactory = groupFactory;
    }

    public Group getGroup(String name) throws EntityException
    {
        DirContext ctx = null;
        NamingEnumeration<SearchResult> enume;
        Group group = null;

        String filter = constructGroupSearchFilter(name).encode();
        String baseDn = searchProperties.getBaseGroupNamespace();

        try
        {
            ctx = repository.getLDAPContext();
            if (log.isDebugEnabled())
                log.debug("AbstractLDAPGroupAdapter.getGroup:" + filter);

            enume = ctx.search(baseDn, filter,
                LDAPUtils.createSearchControls(new String[]{searchProperties.getGroupnameAttribute()}, searchProperties.isGroupSearchScopeAllDepths(),
                    searchProperties.getTimeLimitMillis()));
            if (enume == null) return null;

            while (enume.hasMoreElements())
            {
                SearchResult result = enume.nextElement();
                Attributes attrs = result.getAttributes();
                group = groupFactory.getGroup(attrs, result.getName());
            }
        }
        catch (NamingException e)
        {
            String msg = "Exception when retrieving LDAP group {0} (base DN: {1}, filter: {2})";
            throw new RepositoryException(MessageFormat.format(msg, name, baseDn, filter), e);
        }
        finally
        {
            try
            {
                if (ctx != null) ctx.close();
            }
            catch (NamingException e)
            {
                log.warn("Failed to close LDAP connection after search for group: " + name, e);
            }
        }

        return group;
    }

    public Pager<Group> getGroups() throws EntityException
    {
        LDAPPagerInfo ldapPagerInfo = getGroupEntries();

        return new LDAPEntityPager<Group>(searchProperties, repository, groupFactory, ldapPagerInfo);
    }

    public LDAPPagerInfo getGroupEntries() throws EntityException
    {
        return getGroupEntries("*");
    }

    public LDAPPagerInfo getGroupEntries(String groupName) throws EntityException
    {
        return getGroupEntries(groupName, null, null);
    }

    public LDAPPagerInfo getGroupEntries(String[] attributesToReturn, Filter additionalSearchFilter)
            throws EntityException
    {
        return getGroupEntries("*", attributesToReturn, additionalSearchFilter);
    }

    /**
     * @param groupName a String holding an RFC 2254 compliant getGroupEntries parameter to AND to
     *                  the {@link com.atlassian.user.impl.ldap.repository.LdapContextFactory#GROUP_SEARCH_FILTER}.
     * @throws RepositoryException
     */
    public LDAPPagerInfo getGroupEntries(String groupName, String[] attributesToReturn, Filter additionalSearchFilter)
            throws RepositoryException
    {
        Filter searchFilter = constructGroupSearchFilter(groupName, additionalSearchFilter);
        return search(searchFilter, attributesToReturn);
    }

    /**
     * @return an {@link LDAPPagerInfo} holding the search query used, the result, and the original base context.
     * @throws RepositoryException
     */
    public LDAPPagerInfo search(Filter searchFilter) throws RepositoryException
    {
        return search(searchFilter, null);
    }

    /**
     * @param attributesToReturn - will default to mapped membership attribute available in the {@link com.atlassian.user.impl.ldap.repository.LdapContextFactory}
     *                           if set to null.
     * @return an {@link LDAPPagerInfo} holding the search query used, the result, and the original base context.
     * @throws RepositoryException
     */
    public LDAPPagerInfo search(Filter filter, String[] attributesToReturn) throws RepositoryException
    {
        if (UtilTimerStack.isActive())
            UtilTimerStack.push(this.getClass().getName() + "_search(" + filter + ")");

        try
        {
            if (attributesToReturn == null)
                attributesToReturn = new String[]{searchProperties.getGroupnameAttribute()};

            Filter groupSearchFilter = filterFactory.getGroupSearchFilter();
            if (filter != null)
                groupSearchFilter = LDAPUtils.makeAndFilter(groupSearchFilter, filter);

            SearchControls ctls = LDAPUtils.createSearchControls(
                attributesToReturn, searchProperties.isGroupSearchScopeAllDepths(),
                searchProperties.getTimeLimitMillis());
            NamingEnumeration<SearchResult> groupSearchEnume = null;

            if (UtilTimerStack.isActive())
                UtilTimerStack.push(this.getClass().getName() + "_search_JNDI_RAW_(" + groupSearchFilter + ")");
            DirContext ctx = null;
            try
            {
                ctx = repository.getLDAPContext();

                log.debug("Searching for groups using base name space:" + searchProperties.getBaseGroupNamespace() +
                    " and encoded filter " + groupSearchFilter.encode());

                groupSearchEnume = ctx.search(searchProperties.getBaseGroupNamespace(), groupSearchFilter.encode(), ctls);

                if (groupSearchEnume.hasMore())
                    log.debug("found at least one group");
                else
                    log.debug("no groups found");
            }
            catch (NamingException e)
            {
                throw new RepositoryException(e);
            }
            finally
            {
                if (UtilTimerStack.isActive())
                    UtilTimerStack.pop(this.getClass().getName() + "_search_JNDI_RAW_(" + groupSearchFilter + ")");

                try
                {
                    if (ctx != null) ctx.close();
                }
                catch (NamingException e)
                {
                    log.warn("Exception trying to close LDAP context, possible resource leak", e);
                }
            }

            return new LDAPPagerInfo(groupSearchEnume, groupSearchFilter,
                searchProperties.getBaseGroupNamespace(), searchProperties.isGroupSearchScopeAllDepths(),
                attributesToReturn, searchProperties.getTimeLimitMillis());
        }
        finally
        {
            if (UtilTimerStack.isActive())
                UtilTimerStack.pop(this.getClass().getName() + "_search(" + filter + ")");
        }
    }

    protected Filter constructGroupSearchFilter(String name)
    {
        return constructGroupSearchFilter(name, null);
    }

    protected Filter constructGroupSearchFilter(String name, Filter patternToAnd)
    {
        Filter searchFilter = null;

        if (TextUtils.stringSet(name) && !"*".equals(name)) // if we are only searching for *, then we can exclude the group name attribute from the search
            searchFilter = new EqualsFilter(searchProperties.getGroupnameAttribute(), name);

        searchFilter = addGroupSearchFilter(searchFilter);

        if (patternToAnd != null)
            return LDAPUtils.makeAndFilter(searchFilter, patternToAnd);

        return searchFilter;
    }

    private Filter addGroupSearchFilter(Filter searchFilter)
    {
        return LDAPUtils.makeAndFilter(searchFilter, filterFactory.getGroupSearchFilter());
    }

    /**
     * Extracts the username from the first occurrence of the username attribute
     * <br/>
     * For example,
     * <p/>
     * CN=ldapusera,CN=Users,DC=atlassian,DC=private
     * </p>
     * becomes ldapusera
     *
     * @return String representing the username in the DN
     */
    protected String getFirstPhraseFromDN(String dn)
    {
        String[] rdns = dn.split(",");
        String[] firstPhrase = rdns[0].split("=");

        return firstPhrase[1];
    }


    public String getGroupDN(String groupName) throws EntityException
    {
        LDAPPagerInfo ldapPagerInfo = getGroupEntries(groupName, new String[]{"dn"}, null);

        if (!ldapPagerInfo.getNamingEnumeration().hasMoreElements())
            throw new EntityMissingException("Could not get DN for group [" + groupName + "]");

        SearchResult result = (SearchResult) ldapPagerInfo.getNamingEnumeration().nextElement();

        String groupDN = result.getName();
        //we need a DN. The result only returns the full DN if the result was in the base getGroupEntries context
        if (groupDN.indexOf(searchProperties.getBaseGroupNamespace()) == -1)
            groupDN = groupDN + "," + searchProperties.getBaseGroupNamespace();

        return groupDN;
    }

    public String getGroupDN(Group group) throws EntityException
    {
        if (group instanceof LDAPEntity)
        {
            LDAPEntity entity = (LDAPEntity) group;
            return entity.getDistinguishedName();
        }
        else
            throw new IllegalArgumentException("Group is not an LDAPEntity");
    }

    public LDAPGroupFactory getGroupFactory()
    {
        return groupFactory;
    }
}
