package com.atlassian.user.impl.osuser;

import com.atlassian.user.Entity;
import com.atlassian.user.EntityException;
import com.atlassian.user.User;
import com.atlassian.user.UserManager;
import com.atlassian.user.impl.RepositoryException;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.EntityNameAlphaComparator;
import com.atlassian.user.search.page.DefaultPager;
import com.atlassian.user.search.page.Pager;
import com.atlassian.user.util.Assert;
import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.user.ImmutableException;
import com.opensymphony.user.provider.CredentialsProvider;

import java.util.*;

/**
 * An adaptor class for {@link CredentialsProvider} and some of the higher level operations of {@link com.opensymphony.user.UserManager}
 * <p/>
 * The rule is to use the credentialsProvider and/or profileProvider (for propertySets) for most things. Store() operations
 * must be called on the entity itself.
 */
public class OSUUserManager extends OSUEntityManager implements UserManager
{
    private final OSUAccessor accessor;

    public OSUUserManager(RepositoryIdentifier repository, OSUAccessor accessor)
    {
        super(repository);
        this.accessor = accessor;
    }

    public Pager getUsers()
    {
        Collection userNames = accessor.getCredentialsProvider().list();
        SortedSet atlassianUsers = new TreeSet(new EntityNameAlphaComparator());

        for (Iterator iterator = userNames.iterator(); iterator.hasNext();)
        {
            String username = (String) iterator.next();
            OSUUser user = getWrappedOpensymphonyUser(username);
            if (user != null)
                atlassianUsers.add(user);
        }

        return new DefaultPager(atlassianUsers);
    }

    public Pager getUserNames() throws EntityException
    {
        return new DefaultPager(accessor.getCredentialsProvider().list());
    }

    public User getUser(String username)
    {
        return getWrappedOpensymphonyUser(username);
    }

    private OSUUser getWrappedOpensymphonyUser(String username)
    {
        com.opensymphony.user.User opensymphonyUser = getOpensymphonyUser(username);
        return opensymphonyUser == null ? null : new OSUUser(opensymphonyUser);
    }

    private com.opensymphony.user.User getOpensymphonyUser(String username)
    {
        String lcUsername = username.toLowerCase();
        if (accessor.getCredentialsProvider().handles(lcUsername))
            return new com.opensymphony.user.User(username.toLowerCase(), accessor);
        else
            return null;
    }

    public User createUser(String username) throws com.atlassian.user.EntityException
    {
        boolean usernameHandled = accessor.getCredentialsProvider().handles(username);

        if (usernameHandled)
            throw new com.atlassian.user.impl.DuplicateEntityException("User [" + username
                + "] already exists in credentialsProvider [" + accessor.getCredentialsProvider().toString() + "]");

        if (accessor.getCredentialsProvider().create(username))
        {
            accessor.getProfileProvider().create(username);
            return getWrappedOpensymphonyUser(username);
        }

        //this is a very poor condition to run into but what can be done about it?
        throw new EntityException("Was unable to create user [" + username + "] but the credentials provider ["
            + accessor.getCredentialsProvider().toString() + "] did not say why.");
    }

    /**
     * Encrypts the plain password, sets it on the user, and saves the user.
     */
    public void alterPassword(User user, String plainTextPass) throws EntityException
    {
        user.setPassword(plainTextPass);
        saveUser(user);
    }

    public void removeUser(User user) throws EntityException
    {
        Assert.notNull(user, "User must not be null");
        Assert.isInstanceOf(OSUUser.class, user);
        Assert.isTrue(getUser(user.getName()) != null,
            "User is not managed by this user manager: [" + user.getName() + "]");

        String userName = user.getName();

        // need to remove properties from a user as its deleted
        // See: http://jira.atlassian.com/browse/USER-192
        PropertySet propertySet = accessor.getProfileProvider().getPropertySet(userName);
        for (Iterator i = propertySet.getKeys().iterator(); i.hasNext();)
        {
            String key = (String) i.next();
            propertySet.remove(key);
        }

        List groupsOfUser = accessor.getAccessProvider().listGroupsContainingUser(userName);
        groupsOfUser = new ArrayList(groupsOfUser);

        for (int i = 0; i < groupsOfUser.size(); i++)
        {
            String groupName = (String) groupsOfUser.get(i);
            accessor.getAccessProvider().removeFromGroup(userName, groupName);
        }

        boolean result = accessor.getCredentialsProvider().remove(userName);

        if (!result)
            throw new EntityException("Could not remove user!");

        accessor.getProfileProvider().remove(userName);
    }

    public boolean isReadOnly(User user)
    {
        return getUser(user.getName()) == null;
    }

    public void saveUser(User user) throws EntityException
    {
        Assert.notNull(user, "User must not be null");
        com.opensymphony.user.User osUser = getOpensymphonyUser(user.getName());
        osUser.setEmail(user.getEmail());

        try
        {
            osUser.store();
        }
        catch (ImmutableException e)
        {
            throw new RepositoryException(e);
        }
    }

    public RepositoryIdentifier getRepository(Entity entity) throws EntityException
    {
        return getUser(entity.getName()) == null ? null : repository;
    }

    public OSUAccessor getAccessor()
    {
        return accessor;
    }
}
