package com.atlassian.user.impl.memory;

import com.atlassian.user.Entity;
import com.atlassian.user.EntityException;
import com.atlassian.user.User;
import com.atlassian.user.UserManager;
import com.atlassian.user.impl.DefaultUser;
import com.atlassian.user.impl.DuplicateEntityException;
import com.atlassian.user.impl.memory.provider.MemoryProvider;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.page.Pager;
import com.atlassian.user.security.password.PasswordEncryptor;
import com.atlassian.user.util.Assert;

public class MemoryUserManager implements UserManager
{
    private final MemoryProvider provider;
    private final RepositoryIdentifier repository;
    private final PasswordEncryptor passwordEncryptor;

    public MemoryUserManager(RepositoryIdentifier repository, MemoryProvider provider, PasswordEncryptor passwordEncryptor)
    {
        this.provider = provider;
        this.repository = repository;
        this.passwordEncryptor = passwordEncryptor;
    }

    public Pager getUsers()
    {
        return provider.getUsers();
    }

    public Pager getUserNames() throws EntityException
    {
        return provider.getUserNames();
    }

    public User getUser(String username)
    {
        return provider.getUser(username);
    }

    public User createUser(String username) throws EntityException
    {
        User existingUser = provider.getUser(username);
        if (existingUser != null)
            throw new DuplicateEntityException("User already exists: [" + existingUser + "] in " + getRepository(existingUser));

        User user = new DefaultUser(username);
        provider.addUser(user);

        return user;
    }

    /**
     * Encrypts the plain password, sets it on the user, and saves the user.
     */
    public void alterPassword(User user, String plainTextPass) throws EntityException
    {
        user.setPassword(passwordEncryptor.encrypt(plainTextPass));
    }

    public void removeUser(com.atlassian.user.User user) throws EntityException
    {
        if(provider.getUser(user.getName())==null) {
            throw new IllegalArgumentException("User can not be found in this user manager: [" + user + "]");
        }
        provider.removeUser(user);
    }

    public boolean isReadOnly(User user)
    {
        return getUser(user.getName()) == null;
    }

    public void saveUser(User user) throws EntityException
    {
        Assert.notNull(user, "User must not be null");
        // do nothing -- a memory user doesn't require saving.
    }

    /**
     * @return the {@link RepositoryIdentifier} which is managed by this instance.
     */
    public RepositoryIdentifier getIdentifier()
    {
        return repository;
    }

    public RepositoryIdentifier getRepository(Entity entity) throws EntityException
    {
        if (getUser(entity.getName()) == null)
            return null;

        return repository;
    }

    public boolean isCreative()
    {
        return true;
    }
}
