package com.atlassian.user.impl.ldap.adaptor;

/**
 * Implementations of this adaptor will allow groups to be retrieved
 * from LDAP systems, each with their own different way of structuring groups.
 * <p>
 *
 * When constructing the actual {@link Group} object it is appropriate to use
 * a {@link com.atlassian.user.impl.ldap.LDAPGroupFactory} implementation. Both adaptors
 * and factories are configurable, so they can be replaced at any time by another implementation.
 * <p>
 *
 * Doing this will allow implementors to pull group information from LDAP systems and then
 * construct them into {@link Group} objects however they wish.
 */

import com.atlassian.user.EntityException;
import com.atlassian.user.Group;
import com.atlassian.user.User;
import com.atlassian.user.impl.RepositoryException;
import com.atlassian.user.impl.ldap.search.LDAPPagerInfo;
import com.atlassian.user.search.page.Pager;
import net.sf.ldaptemplate.support.filter.Filter;

public interface LDAPGroupAdaptor
{
    Group getGroup(String name) throws EntityException;

    /**
     * @return A {@link Pager} holding containing a {@link Group} for each group the user belongs to. An empty pager
     * will be returned if the user does not belong to any groups that this manager knows about.
     */
    Pager getGroups(User user) throws EntityException;

    Pager getGroups() throws EntityException;

    Pager findMembers(Group group) throws EntityException;

    Pager findMemberNames(Group group) throws EntityException;

    boolean hasStaticGroups();

    String getGroupDN(Group group) throws EntityException;

    String getGroupDN(String group) throws EntityException;

    boolean hasMembership(Group group, User user) throws EntityException;

    LDAPPagerInfo getGroupEntries() throws EntityException;

    /**
     * If groupName is <code>null</code>, returns all groups.
     */
    LDAPPagerInfo getGroupEntries(String groupName) throws EntityException;

    LDAPPagerInfo getGroupEntries(String[] attributesToReturn, Filter additionalSearchFilter) throws EntityException;

    LDAPPagerInfo getGroupEntries(String groupName, String[] attributesToReturn, Filter additionalSearchFilter)
            throws EntityException;

    LDAPPagerInfo getGroupEntriesViaMembership(User user) throws EntityException;

    LDAPPagerInfo getGroupEntriesViaMembership(String username) throws EntityException;

    LDAPPagerInfo getGroupEntriesViaMembership(String username, String[] attributesToReturn) throws EntityException;

    /**
     * @return an {@link com.atlassian.user.impl.ldap.search.LDAPPagerInfo} holding the search query used, the result, and the original base context.
     * @throws com.atlassian.user.impl.RepositoryException
     */
    LDAPPagerInfo search(Filter searchFilter) throws RepositoryException;
}
