package com.atlassian.user.impl.delegation.security.authentication;

import com.atlassian.user.EntityException;
import com.atlassian.user.UserManager;
import com.atlassian.user.impl.delegation.repository.DelegatingRepository;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.security.authentication.Authenticator;
import com.atlassian.util.profiling.UtilTimerStack;
import org.apache.log4j.Logger;

import java.util.List;
import java.util.Iterator;
import java.util.LinkedList;

public class DelegatingAuthenticator implements Authenticator
{
    private static final Logger log = Logger.getLogger(DelegatingAuthenticator.class);
    private final List authenticators;
    private final UserManager userManager;

    public DelegatingAuthenticator(UserManager userManager, List authenticators)
    {
        this.userManager = userManager;
        this.authenticators = authenticators;
    }

    public boolean authenticate(String username, String password) throws EntityException
    {
        if (UtilTimerStack.isActive())
            UtilTimerStack.push(this.getClass().getName() + "_delegating_authenticate(" + username + ")");

        try
        {
            RepositoryIdentifier repository = userManager.getRepository(userManager.getUser(username));
            List authenticators = getAuthenticatorsForRepository(repository);

            if (authenticators.isEmpty())
            {
                log.error("Failed to find authenticator for user " + username + " from repository " + repository);
                return false;
            }

            for (Iterator it = authenticators.iterator(); it.hasNext();)
            {
                Authenticator authenticator = (Authenticator) it.next();
                if (authenticate(authenticator, username, password))
                    return true;
            }
        }
        finally
        {
            if (UtilTimerStack.isActive())
                UtilTimerStack.pop(this.getClass().getName() + "_delegating_authenticate(" + username + ")");
        }

        return false;
    }

    private boolean authenticate(Authenticator authenticator, String username, String password)
    {
        try
        {
            return authenticator.authenticate(username, password);
        }
        catch (EntityException e)
        {
            log.error(authenticator.getRepository() + ": " + e.getMessage());
            return false;
        }
    }

    private List getAuthenticatorsForRepository(RepositoryIdentifier repository)
    {
        List result = new LinkedList();
        for (Iterator it = authenticators.iterator(); it.hasNext();)
        {
            Authenticator authenticator = (Authenticator) it.next();
            if (authenticator.getRepository().equals(repository))
                result.add(authenticator);
        }
        return result;
    }

    public RepositoryIdentifier getRepository()
    {
        return new DelegatingRepository(authenticators);
    }

    public List getAuthenticators()
    {
        return authenticators;
    }

}
