package com.atlassian.user.impl.delegation;

import com.atlassian.user.Entity;
import com.atlassian.user.EntityException;
import com.atlassian.user.User;
import com.atlassian.user.UserManager;
import com.atlassian.user.impl.DuplicateEntityException;
import com.atlassian.user.impl.delegation.repository.DelegatingRepository;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.page.Pager;
import com.atlassian.user.search.page.PagerFactory;
import com.atlassian.user.util.Assert;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

/*
 * User manager with a list of other user managers to delegate to.
 * The user managers are tried in the iteration order of the list.
 */
public class DelegatingUserManager implements UserManager
{
    private final List userManagers;

    public DelegatingUserManager(List userManagers)
    {
        this.userManagers = userManagers;
    }

    public Pager getUsers() throws EntityException
    {
        ArrayList pagers = new ArrayList();

        Iterator userManagersIter = userManagers.iterator();

        while (userManagersIter.hasNext())
        {
            UserManager userManager = (UserManager) userManagersIter.next();
            pagers.add(userManager.getUsers());
        }

        return PagerFactory.getPager(pagers);
    }

    public Pager getUserNames() throws EntityException
    {
        ArrayList pagers = new ArrayList();
        Iterator userManagersIter = userManagers.iterator();

        while (userManagersIter.hasNext())
        {
            UserManager userManager = (UserManager) userManagersIter.next();
            pagers.add(userManager.getUserNames());
        }

        return PagerFactory.getPager(pagers);
    }

    /**
     * @return - an {@link com.atlassian.user.User} if one could be found, otherwise null.
     * @throws com.atlassian.user.EntityException
     *          - representing the exception which prohibited looking for or
     *          retrieving the user.
     */
    public User getUser(String username) throws EntityException
    {
        User foundUser = null;

        Iterator userManagersIter = userManagers.iterator();

        while (userManagersIter.hasNext())
        {
            UserManager userManager = (UserManager) userManagersIter.next();
            foundUser = userManager.getUser(username);

            if (foundUser != null)
                break;
        }

        return foundUser;
    }

    /**
     * @return the created user
     * @throws com.atlassian.user.EntityException If user cannot be created in any user managers.
     */
    public User createUser(String username) throws EntityException
    {
        User preexistingUser;
        try
        {
            preexistingUser = getUser(username);
        }
        catch (EntityException e)
        {
            throw new EntityException("Couldn't check whether user already exists", e);
        }

        if (preexistingUser != null)
            throw new DuplicateEntityException("User [" + username + "] already exists in " + getRepository(preexistingUser).getName());


        User createdUser = null;
        Iterator userManagersIter = userManagers.iterator();

        while (userManagersIter.hasNext())
        {
            UserManager userManager = (UserManager) userManagersIter.next();

            if (userManager.isCreative())
                createdUser = userManager.createUser(username);

            if (createdUser != null)
                break;
        }

        if (createdUser == null)
            throw new EntityException("Could not create user: " + username + ". " +
                "Ensure you have a read-write repository configured.");

        return createdUser;
    }

    /**
     * Encrypts the plain password, sets it on the user, and saves the user.
     */
    public void alterPassword(User user, String plainTextPass) throws EntityException
    {
        UserManager userManager = getMatchingUserManager(user);

        if (userManager != null)
            userManager.alterPassword(user, plainTextPass);
        else
            throw new EntityException("Cannot find a userManager responsible for user [" + user.getName() + "]");
    }

    public void saveUser(User user) throws EntityException
    {
        UserManager userManager = getMatchingUserManager(user);

        if (userManager != null)
            userManager.saveUser(user);
        else
            throw new EntityException("Cannot find a userManager responsible for user [" + user.getName() + "]");
    }

    public void removeUser(User user) throws EntityException
    {
        UserManager userManager = getMatchingUserManager(user);

        if (userManager != null)
            userManager.removeUser(user);
        else
            throw new IllegalArgumentException("Cannot find a userManager responsible for user [" + user.getName() + "]");
    }

    /**
     * @return true indicates that information on the user object cannot be altered in the storage system
     *         (see {@link com.atlassian.user.repository.RepositoryIdentifier}),
     *         false indicates that the storage system will save changes or that this {@link com.atlassian.user.UserManager} does not
     *         know about the {@link com.atlassian.user.User}.
     * @throws com.atlassian.user.EntityException
     *
     */
    public boolean isReadOnly(User user) throws EntityException
    {
        UserManager userManager = getMatchingUserManager(user);

        if (userManager != null)
            return userManager.isReadOnly(user);
        else
            throw new EntityException("Cannot find a userManager responsible for user [" + user.getName() + "]");
    }

    /**
     * @return the {@link com.atlassian.user.repository.RepositoryIdentifier} which is managed by this instance.
     */
    public RepositoryIdentifier getIdentifier()
    {
        Iterator iter = userManagers.iterator();
        ArrayList repositories = new ArrayList();

        while (iter.hasNext())
        {
            UserManager userManager = (UserManager) iter.next();
            repositories.add(userManager.getIdentifier());
        }

        return new DelegatingRepository(repositories);
    }

    /**
     * @return the {@link com.atlassian.user.repository.RepositoryIdentifier} in which the entity is stored, otherwise null.
     * @throws com.atlassian.user.EntityException
     *
     */
    public RepositoryIdentifier getRepository(Entity entity) throws EntityException
    {
        Iterator iter = userManagers.iterator();

        while (iter.hasNext())
        {
            UserManager userManager = (UserManager) iter.next();
            RepositoryIdentifier repo = userManager.getRepository(entity);

            if (repo != null)
                return repo;
        }

        return null;
    }

    /**
     * Used to detemine whether an entity can be created (eg, can call {@link com.atlassian.user.UserManager#createUser(String)} or
     * {@link com.atlassian.user.GroupManager#createGroup(String)}
     *
     * @return true to indicate that {@link com.atlassian.user.Entity} objects can be created by this manager, or false to indicate
     *         not.
     */
    public boolean isCreative()
    {
        Iterator iter = userManagers.iterator();

        while (iter.hasNext())
        {
            UserManager userManager = (UserManager) iter.next();

            if (userManager.isCreative())
                return true;
        }

        return false;
    }

    /**
     * Helper method to locate the first userManager responsible for the given user, in the delegation.
     */
    protected UserManager getMatchingUserManager(User user) throws EntityException
    {
        Assert.notNull(user, "User must not be null");
        Iterator userManagersIter = userManagers.iterator();

        while (userManagersIter.hasNext())
        {
            UserManager matchingUserManager = (UserManager) userManagersIter.next();
            User foundUser = matchingUserManager.getUser(user.getName());

            if (foundUser != null)
                return matchingUserManager;
        }

        return null;
    }

    /**
     * @return an unmodifiable list of {@link UserManager}s which this manager delegates to,
     * in the order of delegation.
     */
    public List getUserManagers()
    {
        return Collections.unmodifiableList(userManagers);
    }
}
