package com.atlassian.user.impl.delegation;

import com.atlassian.user.*;
import com.atlassian.user.impl.delegation.repository.DelegatingRepository;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.page.DefaultPager;
import com.atlassian.user.search.page.Pager;
import com.atlassian.user.search.page.PagerFactory;
import com.atlassian.user.util.Assert;
import org.apache.log4j.Logger;

import java.util.*;

/*
 * Group manager with a list of other group managers to delegate to.
 * The group managers are tried in the iteration order of the list.
 */
public class DelegatingGroupManager implements GroupManager
{
    private static final Logger log = Logger.getLogger(DelegatingGroupManager.class);

        /** List of {@link GroupManager}s. */
    private final List groupManagers;

    public DelegatingGroupManager(List groupManagers)
    {
        this.groupManagers = groupManagers;
    }

    /**
     * Returns an iterator of {@link GroupManager} objects, in the order they
     * are to be used for this delegating manager.
     */
    protected Iterator getGroupManagersIterator()
    {
        return groupManagers.iterator();
    }

    /**
     * @return an unmodifiable list of {@link GroupManager}s which this manager delegates to,
     * in the order of delegation.
     */
    public List getGroupManagers()
    {
        return Collections.unmodifiableList(groupManagers);
    }

    /**
     * Returns the first group manager in iteration order which returns a non-null value from
     * {@link GroupManager#getGroup(String)}, or the last group manager in the iteration
     * order if no group manager meets this criteria.
     *
     * @throws EntityException if any of the delegating managers throws an exception
     * in {@link GroupManager#getGroup(String)}.
     * @throws NullPointerException if the group is null
     */
    protected GroupManager getMatchingGroupManager(Group group) throws EntityException
    {
        GroupManager groupManager = null;
        Iterator iter = getGroupManagersIterator();

        while (iter.hasNext())
        {
            groupManager = (GroupManager) iter.next();
            Group foundGroup = groupManager.getGroup(group.getName());

            if (foundGroup != null)
                return groupManager;
        }

        // returns the last groupManager if none matched
        return groupManager;
    }

    public Pager getGroups() throws EntityException
    {
        List groups = new ArrayList();

        Iterator iter = getGroupManagersIterator();

        while (iter.hasNext())
        {
            GroupManager groupManager = (GroupManager) iter.next();
            try
            {
                groups.add(groupManager.getGroups());
            }
            catch (EntityException e)
            {
                log.error("Failed to retrieve groups from group manager in delegation: " +
                    groupManager.getClass().toString() +". Continuing with remaining managers.", e);
            }
        }

        return PagerFactory.getPager(groups);
    }

    public Pager getGroups(User user) throws EntityException
    {
        Assert.notNull(user, "User must not be null");
        Iterator iter = getGroupManagersIterator();
		List pagers = new LinkedList();

		while (iter.hasNext())
        {
            GroupManager groupManager = (GroupManager) iter.next();
            try
            {
                Pager groupPager = groupManager.getGroups(user);
                pagers.add(groupPager);
            }
            catch (EntityException e)
            {
                log.error("Failed to retrieve groups for user [" + user + "] from group manager: " +
                    groupManager.getClass().toString() +". Continuing with remaining managers.", e);
            }
        }

		return PagerFactory.getPager(pagers);
    }

    public List/*<Group>*/ getWritableGroups()
    {
        List groups = new ArrayList();

        for (Iterator it = getGroupManagersIterator(); it.hasNext();)
        {
            GroupManager groupManager = (GroupManager) it.next();
            groups.addAll(groupManager.getWritableGroups());
        }

        return groups;
    }

    public Pager getMemberNames(Group group) throws EntityException
    {
        GroupManager groupManager = getMatchingGroupManager(group);

        if (groupManager == null)
            return new DefaultPager();

        return groupManager.getMemberNames(group);
    }

    public Pager getLocalMemberNames(Group group) throws EntityException
    {
        GroupManager groupManager = getMatchingGroupManager(group);

        if (groupManager == null)
            return new DefaultPager();

        return groupManager.getLocalMemberNames(group);
    }

    public Pager getExternalMemberNames(Group group) throws EntityException
    {
        GroupManager groupManager = getMatchingGroupManager(group);

        if (groupManager == null)
            return new DefaultPager();

        return groupManager.getExternalMemberNames(group);
    }

    public Group getGroup(String groupName) throws EntityException
    {
        Iterator iter = getGroupManagersIterator();
        Group foundGroup;

        while (iter.hasNext())
        {
            GroupManager groupManager = (GroupManager) iter.next();

            foundGroup = groupManager.getGroup(groupName);

			if (foundGroup != null)
                return foundGroup;
        }

        return null;
    }

    public Group createGroup(String groupName) throws EntityException
    {
        Iterator iter = getGroupManagersIterator();
        Group createdGroup = null;

        while (iter.hasNext())
        {
            GroupManager groupManager = (GroupManager) iter.next();

            if (groupManager.isCreative())
                createdGroup = groupManager.createGroup(groupName);

            if (createdGroup != null)
                return createdGroup;
        }

        return null;
    }

    /**
     * Removes the specified group, if it is present.
     *
     * @throws EntityException
     *          - representing the exception which prohibited removal
     */
    public void removeGroup(Group group) throws EntityException
    {
        Iterator iter = getGroupManagersIterator();

        while (iter.hasNext())
        {
            GroupManager groupManager = (GroupManager) iter.next();
            if (groupManager.getGroup(group.getName()) == null)
                continue;

            if (groupManager.isReadOnly(group))
                continue;

            groupManager.removeGroup(group);
            break;
        }
    }

    public void addMembership(Group group, User user) throws EntityException
    {
        if (group == null)
            throw new IllegalArgumentException("Can't add membership for null group");
        GroupManager groupManager = getMatchingGroupManager(group);
        groupManager.addMembership(group, user);
    }

    /**
     * Check whether this manager has a record of membership between the
     * argued user and group.
     *
     * @throws com.atlassian.user.impl.RepositoryException
     *
     */
    public boolean hasMembership(Group group, User user) throws EntityException
    {
        if (group == null)
            return false;
        GroupManager groupManager = getMatchingGroupManager(group);
        return groupManager.hasMembership(group, user);
    }

    public void removeMembership(Group group, User user) throws EntityException
    {
        if (group == null)
            throw new IllegalArgumentException("Can't remove membership for null group");
        GroupManager groupManager = getMatchingGroupManager(group);
        groupManager.removeMembership(group, user);
    }

    /**
     * @return true if users from other systems can be granted membership in the underlying {@link com.atlassian.user.repository.RepositoryIdentifier},
     *         otherwise false.
     * @throws EntityException
     *
     */
    public boolean supportsExternalMembership() throws EntityException
    {
        Iterator iter = getGroupManagersIterator();

        while (iter.hasNext())
        {
            GroupManager groupManager = (GroupManager) iter.next();

            if (groupManager.supportsExternalMembership())
                return true;
        }

        return false;
    }

    public boolean isReadOnly(Group group) throws EntityException
    {
        Iterator iter = getGroupManagersIterator();

        while (iter.hasNext())
        {
            GroupManager groupManager = (GroupManager) iter.next();

            if (groupManager.getGroup(group.getName()) != null)
                return groupManager.isReadOnly(group);
        }

        return false;
    }

    /**
     * @return the {@link com.atlassian.user.repository.RepositoryIdentifier} which is managed by this instance.
     */
    public RepositoryIdentifier getIdentifier()
    {
        ArrayList repositories = new ArrayList();

        Iterator iter = getGroupManagersIterator();

        while (iter.hasNext())
        {
            GroupManager groupManager = (GroupManager) iter.next();
            RepositoryIdentifier repo = groupManager.getIdentifier();
            repositories.add(repo);
        }

        return new DelegatingRepository(repositories);
    }

    /**
     * Don't use for now. This is very expensive for large LDAP instances. todo: get user/group entities to store which repository they came from
     *
     * @return the {@link RepositoryIdentifier} in which the entity is stored, otherwise null.
     * @throws EntityException
     *
     */
    public RepositoryIdentifier getRepository(Entity entity) throws EntityException
    {
        if (!(entity instanceof Group))
            return null;

        GroupManager groupManager = getMatchingGroupManager((Group) entity);

        return groupManager.getIdentifier();
    }

    /**
     * Used to detemine whether an entity can be created (eg, can call {@link UserManager#createUser(String)} or
     * {@link GroupManager#createGroup(String)}
     *
     * @return true to indicate that {@link Entity} objects can be created by this manager, or false to indicate
     *         not.
     */
    public boolean isCreative()
    {
        Iterator iter = getGroupManagersIterator();

        while (iter.hasNext())
        {
            GroupManager groupManager = (GroupManager) iter.next();

            if (groupManager.isCreative())
                return true;
        }
        return false;
    }
}
