package com.atlassian.user;

import com.atlassian.user.search.page.Pager;
import com.atlassian.user.impl.DuplicateEntityException;

import java.util.List;

/**
 * A group manager is responsible for all CRUD operations on groups and group
 * memberships in a given repository.
 *
 * Where an operation requires a {@link Group} as an argument, calling the
 * method with a group from another repository will typically throw an
 * {@link IllegalArgumentException}.
 *
 * The group manager is responsible for ensuring the consistency of the underlying
 * repository where relevant. For example, if membership information needs to be
 * updated as part of the {@link #removeGroup(Group)} operation, the group manager
 * is responsible for ensuring it happens.
 *
 * This interface is an extension of the {@link EntityManager} interface which
 * is common to user and group management. The common functionality includes
 * initialisation and access to the repository description.
 */
public interface GroupManager extends EntityManager
{
    /**
     * Retrieves all groups in this repository.
     *
     * @return a {@link Pager} containing a {@link Group} for each group managed
     * by the repository. An empty pager will be returned if the repository does
     * not contain any groups.
     */
    Pager/*<Group>*/ getGroups() throws EntityException;

    /**
     * Retrieves the groups to which the given user belongs. Only groups which
     * are managed by this repository are included.
     *
     * @return a {@link Pager} containing a {@link Group} for each group the user
     * belongs to. An empty pager will be returned if the user does not belong to
     * any groups that this manager knows about.
     * @throws IllegalArgumentException if the user provided is <tt>null</tt>
     */
    Pager/*<Group>*/ getGroups(User user) throws EntityException;

    /**
     * Gets all editable groups.  That is, returns any groups that belong to read/write repositories. Groups in readonly
     * repositories that are ReadOnly are not returned.
     *
     * @return list of {@link Group}s that can be edited.
     */
    List /*<Group>*/ getWritableGroups();

    /**
     * Retrieves the names of all members of the specified group. The names are
     * ordered by a Collator for the JVM's default locale (as returned by
     * {@link java.text.Collator#getInstance()}.
     *
     * @return a {@link Pager} containing a {@link String} with the name of each
     * member of the group. An empty pager will be returned if the group has no
     * members.
     */
    Pager/*<String>*/ getMemberNames(Group group) throws EntityException;

    /**
     * Retrieves the names of those members of the specified group which are also
     * stored in this repository. The names are ordered by a Collator for the JVM's
     * default locale (as returned by {@link java.text.Collator#getInstance()}.
     * <p/>
     * If {@link #supportsExternalMembership()} returns <code>false</code>, this
     * call is equivalent to {@link #getMemberNames(Group)}.
     *
     * @return a {@link Pager} containing a {@link String} with the name of each
     * member of the group which is stored in the same repository. An empty pager
     * will be returned if the group has no local members.
     */
    Pager/*<String>*/ getLocalMemberNames(Group group) throws EntityException;

    /**
     * Retrieves the names of those members of the specified group which are
     * stored in another repository. For example, an LDAP user which is a member
     * of a Hibernate group is an external member of the Hibernate group.
     * <p/>
     * The names are ordered by a Collator for the JVM's default locale (as
     * returned by {@link java.text.Collator#getInstance()}.
     * <p/>
     * If {@link #supportsExternalMembership()} returns <code>false</code>, this
     * method throws {@link UnsupportedOperationException}.
     *
     * @return a {@link Pager} containing a {@link String} with the name of each
     * member of the group which is stored in a different repository. An empty pager
     * will be returned if the group has no external members.
     * @throws UnsupportedOperationException if {@link #supportsExternalMembership()}
     * returns <code>false</code>.
     */
    Pager/*<String>*/ getExternalMemberNames(Group group) throws EntityException;

    /**
     * Retrieves the group with the given name. Returns <code>null</code> if the
     * group does not exist in this repository.
     *
     * @return a {@link Group} or null if the group does not exist.
     */
    Group getGroup(String groupName) throws EntityException;

    /**
     * Create a new group with the specified name.
     *
     * @throws DuplicateEntityException if a group with the provided name already exists.
     * @throws EntityException if the {@link Group} could not be created.
     * @throws UnsupportedOperationException if {@link #isCreative()} returns false
     */
    Group createGroup(String groupName) throws EntityException;

    /**
     * Remove the group specified, if it exists in this repository. If the group does not belong
     * to this repository, an {@link IllegalArgumentException} will be thrown.
     *
     * If required to maintain the consistency of the repository, the group manager
     * should remove users from the group before removing the group itself.
     *
     * @throws EntityException if the {@link Group} could not be removed.
     * @throws UnsupportedOperationException if {@link #isReadOnly(Group)} returns true.
     */
    void removeGroup(Group group) throws EntityException;

    /**
     * Adds the user to the specified group.
     * <p/>
     * If the user is not in this repository, and {@link #supportsExternalMembership()}
     * returns <code>true</code>, the user will be added as an external user.
     *
     * @throws UnsupportedOperationException if {@link #isReadOnly(Group)} returns true.
     * @throws IllegalArgumentException if the group is not handled by this group manager
     * or the group is <tt>null</tt>.
     */
    void addMembership(Group group, User user)  throws EntityException, IllegalArgumentException;

    /**
     * Returns true if the user is a member of the specified group.
     * <p/>
     * If the user is not in this repository, and {@link #supportsExternalMembership()}
     * returns <code>true</code>, external members will be checked as well.
     * <p/>
     * If the group is not handled by this manager, returns false.

     * @return true if the user is a member of the specified group, otherwise false.
     */
    boolean hasMembership(Group group, User user) throws EntityException;

    /**
     * Removes the user from the specified group.
     *
     * @throws EntityException if the membership could not be removed.
     * @throws UnsupportedOperationException if {@link GroupManager#isReadOnly(Group)}
     * returns true.
     * @throws IllegalArgumentException if the group is not handled by this manager
     */
    void removeMembership(Group group, User user) throws EntityException, IllegalArgumentException;

    /**
     * Returns <code>true</code> if the repository supports users in other
     * repositories being members of groups in this repository.
     * <p/>
     * Typically this is true of an application-specific Hibernate repository, but
     * not of a company's LDAP server. It is designed to allow the LDAP users to
     * be members of the application's groups for flexible application-level
     * security.
     *
     * @return true if users from other repositories can be granted membership
     * to groups in this repository, otherwise false.
     */
    boolean supportsExternalMembership() throws EntityException;

    /**
     * Returns <code>true</code> if the specified group and membership of the
     * specified group cannot be modified in the repository.
     * <p/>
     * If this returns <code>true</code>, invoking methods which attempt to
     * modify the group or membership of the group will fail with
     * {@link UnsupportedOperationException}.
     *
     * @return true if the group and membership of the group cannot be modified,
     * otherwise false.
     */
    boolean isReadOnly(Group group) throws EntityException;
}
