package com.atlassian.user.util.migration;

import com.atlassian.user.*;
import com.atlassian.user.configuration.RepositoryAccessor;
import com.atlassian.user.configuration.DelegationAccessor;
import com.atlassian.user.configuration.DefaultDelegationAccessor;
import com.atlassian.user.impl.DefaultUser;
import com.atlassian.user.impl.RepositoryException;
import com.atlassian.user.impl.hibernate.DefaultHibernateUser;
import com.atlassian.user.impl.hibernate.properties.HibernatePropertySetFactory;
import com.atlassian.user.impl.osuser.OSUAccessor;
import com.atlassian.user.impl.osuser.OSUUserManager;
import com.opensymphony.user.provider.AccessProvider;
import net.sf.hibernate.HibernateException;
import net.sf.hibernate.Session;
import net.sf.hibernate.SessionFactory;
import org.apache.log4j.Logger;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowCallbackHandler;
import org.springframework.jdbc.datasource.SingleConnectionDataSource;
import org.springframework.orm.hibernate.SessionFactoryUtils;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.*;

/**
 * Makes a raw JDBC connection to os_user tables and copies across information into the supplied {@link
 * com.atlassian.user.UserManager}, {@link com.atlassian.user.GroupManager}, and {@link
 * com.atlassian.user.properties.PropertySetFactory}
 */

public class OSUEntityMigrator implements EntityMigrator
{
    private static final Logger log = Logger.getLogger(OSUEntityMigrator.class);

    private static final String OSUSER_REPOSITORY_KEY = "osuserRepository";

    private UserManager targetUserManager;
    private GroupManager targetGroupManager;

    private AccessProvider osAccessProvider;

    private final SessionFactory sessionFactory;

    public OSUEntityMigrator(RepositoryAccessor osuserRepositoryAccessor, RepositoryAccessor repositoryAccessor, SessionFactory sessionFactory)
    {
        if (osuserRepositoryAccessor == null)
            throw new IllegalArgumentException("osuserRepositoryAccessor is required.");
        if (repositoryAccessor == null)
            throw new IllegalArgumentException("targetRepositoryAccessor is required.");
        if (sessionFactory == null)
            throw new IllegalArgumentException("sessionFactory is required.");

        this.sessionFactory = sessionFactory;

        final DelegationAccessor targetRepositoryAccessor = getNonOSUserRepositoryAccessor(repositoryAccessor);
        if (!targetRepositoryAccessor.getRepositoryAccessors().isEmpty())
        {
            final UserManager osUserManager = osuserRepositoryAccessor.getUserManager();
            if (osUserManager == null)
                throw new IllegalArgumentException("osUserManager is required.");

            final OSUAccessor osuAccessor = ((OSUUserManager) osUserManager).getAccessor();
            if (osuAccessor == null)
                throw new IllegalArgumentException("osuAccessor is required.");

            osAccessProvider = osuAccessor.getAccessProvider();
            if (osAccessProvider == null)
                throw new IllegalArgumentException("osAccessProvider is required.");

            targetUserManager = targetRepositoryAccessor.getUserManager();
            targetGroupManager = targetRepositoryAccessor.getGroupManager();

            if (targetUserManager == null)
                throw new IllegalArgumentException("userManager is required.");
            if (targetGroupManager == null)
                throw new IllegalArgumentException("groupManager is required.");
        }
    }

    private DelegationAccessor getNonOSUserRepositoryAccessor(RepositoryAccessor repositoryAccessor)
    {
        final DelegationAccessor nonOSUserDelegationAccessor = new DefaultDelegationAccessor();
        if (repositoryAccessor instanceof DelegationAccessor)
        {
            final DelegationAccessor delegationAccessor = (DelegationAccessor) repositoryAccessor;
            for (Iterator iterator = delegationAccessor.getRepositoryAccessors().iterator(); iterator.hasNext();)
            {
                final RepositoryAccessor accessor = (RepositoryAccessor) iterator.next();
                if (!OSUSER_REPOSITORY_KEY.equals(accessor.getIdentifier().getKey()))
                    nonOSUserDelegationAccessor.addRepositoryAccessor(accessor);
            }
            return nonOSUserDelegationAccessor;
        }
        else
        {
            if (!OSUSER_REPOSITORY_KEY.equals(repositoryAccessor.getIdentifier().getKey()))
                nonOSUserDelegationAccessor.addRepositoryAccessor(repositoryAccessor);
        }
        return nonOSUserDelegationAccessor;
    }

    /**
     * Must be done with raw JDBC to access the password value.
     *
     * @param config the migrator configuration
     * @throws RepositoryException if there is no non OSUser repository was included in target repository Accessor
     */
    public void migrate(MigratorConfiguration config, MigrationProgressListener progressListener) throws EntityException
    {
        if (targetUserManager == null)
        {
            throw new RepositoryException("No non OSUser repository configured. Cannot perform migration.");
        }
        final DataSource dataSource = getDataSource();
        final OSUserDao osUserDao = new OSUserDao(dataSource);
        final Map users = osUserDao.findAllUsers();


        final Collection existingUsernames = new HashSet();
        final Collection nonExistingUsernames = new HashSet();

        // starting user migration
        progressListener.userMigrationStarted(users.size());

        for (Iterator it = users.entrySet().iterator(); it.hasNext();)
        {
            Map.Entry userEntry = (Map.Entry) it.next();
            final Long osUserId = (Long) userEntry.getKey();
            final User user = (User) userEntry.getValue();
            if (userExists(targetUserManager, user))
            {
                existingUsernames.add(user.getName());
            }
            else
            {
                nonExistingUsernames.add(user.getName());
                addUser(targetUserManager, (DefaultUser) user);
            }

            try
            {
                migratePropertySet(osUserId, user);
            }
            catch (EntityException e)
            {
                log.error("Error migrating property set for user: " + user + " with id: " + osUserId, e);
            }

            progressListener.userMigrated();
        }

        // users migrated, starting group migration
        progressListener.userMigrationComplete();

        final List groups = osAccessProvider.list();
        progressListener.groupMigrationStarted(groups.size());

        for (Iterator groupsIterator = groups.iterator(); groupsIterator.hasNext();)
        {
            final String groupName = (String) groupsIterator.next();
            final Group migratedGroup = getOrCreateGroup(groupName);

            final List members = osAccessProvider.listUsersInGroup(groupName);
            for (Iterator membersIterator = members.iterator(); membersIterator.hasNext();)
            {
                final String member = (String) membersIterator.next();
                if (log.isInfoEnabled()) log.info("Adding member <" + member + "> to group <" + groupName + ">");


                if (nonExistingUsernames.contains(member) || (existingUsernames.contains(member) && config.isMigrateMembershipsForExistingUsers()))
                {
                    targetGroupManager.addMembership(migratedGroup, targetUserManager.getUser(member));
                }
            }
            progressListener.groupMigrated();
        }

        // group migration complete
        progressListener.groupMigrationComplete();
    }

    private void migratePropertySet(Long userId, User user) throws EntityException
    {
        if (log.isInfoEnabled()) log.info("Migrating properties for <" + user.getName() + ">");

        final User targetUser = targetUserManager.getUser(user.getName());
        final String osUserId = String.valueOf(userId);
        final String entityName = getEntityName(targetUser);
        final String entityId = getEntityId(targetUser);

        final JdbcTemplate template = new JdbcTemplate(getDataSource());
        template.query("SELECT * FROM OS_PROPERTYENTRY WHERE ENTITY_NAME = 'OSUser_user' AND ENTITY_ID = ? AND ENTITY_KEY <> 'fullName' AND ENTITY_KEY <> 'email'", new String[]{osUserId}, new RowCallbackHandler()
        {
            public void processRow(ResultSet resultSet) throws SQLException
            {
                template.update("INSERT INTO OS_PROPERTYENTRY (ENTITY_NAME,ENTITY_ID,ENTITY_KEY,KEY_TYPE,BOOLEAN_VAL,DOUBLE_VAL,STRING_VAL,LONG_VAL,INT_VAL,DATE_VAL) VALUES (?,?,?,?,?,?,?,?,?,?)", new Object[]{
                    entityName,
                    entityId,
                    resultSet.getString("ENTITY_KEY"),
                    new Integer(resultSet.getInt("KEY_TYPE")),
                    Boolean.valueOf(resultSet.getBoolean("BOOLEAN_VAL")),
                    new Double(resultSet.getDouble("DOUBLE_VAL")),
                    resultSet.getString("STRING_VAL"),
                    new Long(resultSet.getLong("LONG_VAL")),
                    new Integer(resultSet.getInt("INT_VAL")),
                    resultSet.getTimestamp("DATE_VAL")
                });
            }
        });
    }

    private String getEntityName(User user) throws EntityException
    {
        if (user instanceof DefaultHibernateUser)
        {
            return HibernatePropertySetFactory.LOCAL_USER + "_" + user.getName();
        }
        else if (user instanceof ExternalEntity)
        {
            return HibernatePropertySetFactory.EXTERNAL_ENTITY + "_" + user.getName();
        }
        else
        {
            throw new EntityException("Could not determine entityName for user: " + user + " of type: " + (user != null ? user.getClass().getName() : "unkown"));
        }
    }

    private String getEntityId(User user) throws EntityException
    {
        if (user instanceof DefaultHibernateUser)
        {
            return String.valueOf(((DefaultHibernateUser) user).getId());
        }
        else if (user instanceof ExternalEntity)
        {
            return String.valueOf(((ExternalEntity) user).getId());
        }
        else
        {
            throw new EntityException("Could not find id for user: " + user + " of type: " + (user != null ? user.getClass().getName() : "unkown"));
        }
    }

    /**
     * Adds the given user using the user manager.
     *
     * @param userManager the user manager to add the user to
     * @param user the user to add
     * @throws EntityException if a problem occur adding the user in the user manager
     */
    private void addUser(UserManager userManager, DefaultUser user) throws EntityException
    {
        if (log.isInfoEnabled()) log.info("Adding user <" + user.getName() + ">");

        final User newUser = userManager.createUser(user.getName());
        newUser.setFullName(user.getFullName());
        newUser.setEmail(user.getEmail());
        newUser.setPassword(user.getPassword());
        userManager.saveUser(newUser);
    }

    private Group getOrCreateGroup(String groupName) throws EntityException
    {
        Group group = targetGroupManager.getGroup(groupName);
        if (group == null)
        {
            if (log.isInfoEnabled()) log.info("Creating group <" + groupName + ">");
            group = targetGroupManager.createGroup(groupName);
        }

        return group;
    }

    private boolean userExists(UserManager userManager, User user) throws EntityException
    {
        return user != null && userManager.getUser(user.getName()) != null;
    }

    private DataSource getDataSource()
    {
        Session hibernateSession = SessionFactoryUtils.getSession(sessionFactory, true);
        Connection conn;
        try
        {
            conn = hibernateSession.connection();
        }
        catch (HibernateException e)
        {
            throw new RuntimeException(e);
        }
        return new SingleConnectionDataSource(conn, true);
    }

}
