package com.atlassian.user.impl.ldap;

import com.atlassian.user.Entity;
import com.atlassian.user.EntityException;
import com.atlassian.user.Group;
import com.atlassian.user.User;
import com.atlassian.user.impl.ReadOnlyGroupManager;
import com.atlassian.user.impl.ldap.adaptor.LDAPGroupAdaptor;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.page.DefaultPager;
import com.atlassian.user.search.page.Pager;

public class LDAPGroupManagerReadOnly extends ReadOnlyGroupManager
{
    private final RepositoryIdentifier repositoryIdentifier;
    private final LDAPGroupAdaptor groupAdaptor;

    public LDAPGroupManagerReadOnly(RepositoryIdentifier repositoryIdentifier, LDAPGroupAdaptor groupAdaptor)
    {
        this.repositoryIdentifier = repositoryIdentifier;
        this.groupAdaptor = groupAdaptor;
    }

    /**
     * @return the {@link com.atlassian.user.repository.RepositoryIdentifier} which is managed by this instance.
     */
    public RepositoryIdentifier getIdentifier()
    {
        return repositoryIdentifier;
    }

    public RepositoryIdentifier getRepository(Entity entity) throws EntityException
    {
        if (getGroup(entity.getName()) == null) // not handled by this manager
            return null;

        return repositoryIdentifier;

    }

    public Pager getGroups() throws EntityException
    {
        return groupAdaptor.getGroups();
    }

    public Group getGroup(String groupName) throws EntityException
    {
        return groupAdaptor.getGroup(groupName);
    }

    public Pager getGroups(User user) throws EntityException
    {
        if (!LDAPValidator.validateLDAPEntity(user))
            return new DefaultPager();

        return groupAdaptor.getGroups(user);
    }

    public Pager getMemberNames(Group group) throws EntityException
    {
        if (!LDAPValidator.validateLDAPEntity(group))
            return new DefaultPager();

        return groupAdaptor.findMemberNames(group);
    }

    public Pager getLocalMemberNames(Group group) throws EntityException
    {
        if (!LDAPValidator.validateLDAPEntity(group))
            return new DefaultPager();

        return groupAdaptor.findMembers(group);
    }

    public Pager getExternalMemberNames(Group group) throws EntityException
    {
        throw new UnsupportedOperationException("External membership is not supported.");
    }

    public boolean hasMembership(Group group, User user) throws EntityException
    {
        return groupAdaptor.hasMembership(group, user);
    }

    public boolean supportsExternalMembership() throws EntityException
    {
        return false;
    }
}