package com.atlassian.user.impl.ldap;

import com.atlassian.user.Entity;
import com.atlassian.user.EntityException;
import com.atlassian.user.Group;
import com.atlassian.user.impl.RepositoryException;
import com.atlassian.user.impl.ldap.properties.LdapMembershipProperties;
import com.atlassian.user.impl.ldap.properties.LdapSearchProperties;
import com.atlassian.util.profiling.UtilTimerStack;

import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;

public class DefaultLDAPGroupFactory implements LDAPGroupFactory
{
    private final LdapSearchProperties searchProperties;
    private final LdapMembershipProperties membershipProperties;

    public DefaultLDAPGroupFactory(LdapSearchProperties searchProperties,
        LdapMembershipProperties membershipProperties)
    {
        this.membershipProperties = membershipProperties;
        this.searchProperties = searchProperties;
    }

    public Group getGroup(Attributes attrs, String distinguishedName) throws EntityException
    {
        if (UtilTimerStack.isActive())
            UtilTimerStack.push(this.getClass().getName() + "_delegating_getGroup(attrs, " + distinguishedName + ")");

        Group group;

        try
        {
            Attribute groupNameAttribute = attrs.get(searchProperties.getGroupnameAttribute());
            if (groupNameAttribute != null)
            {
                String groupName = (String) groupNameAttribute.get();
                group = new DefaultLDAPGroup(groupName, distinguishedName);
            }
            else
            {
                Attribute membershipAttribute = attrs.get(membershipProperties.getMembershipAttribute());
                group = getGroup(membershipAttribute);
            }

        }
        catch (NamingException e)
        {
            throw new RepositoryException(e);
        }

        if (UtilTimerStack.isActive())
            UtilTimerStack.pop(this.getClass().getName() + "_delegating_getGroup(attrs, " + distinguishedName + ")");

        return group;
    }

    public Group getGroup(String distinguishedName)
    {
        if (UtilTimerStack.isActive())
            UtilTimerStack.push(this.getClass().getName() + "_delegating_getGroup(" + distinguishedName + ")");

        String groupName = getGroupNameFromDN(distinguishedName);
        Group group = new DefaultLDAPGroup(groupName, distinguishedName);

        if (UtilTimerStack.isActive())
            UtilTimerStack.pop(this.getClass().getName() + "_delegating_getGroup(" + distinguishedName + ")");

        return group;
    }

    public Group getGroup(Attribute attribute) throws EntityException
    {
        if (UtilTimerStack.isActive())
            UtilTimerStack.push(this.getClass().getName() + "_delegating_getGroup(attr)");

        try
        {
            String groupDN = getGroupDNFromMembershipAttribute(attribute);
            String groupName = getGroupNameFromDN(groupDN);

            return new DefaultLDAPGroup(groupName, groupDN);
        }
        finally
        {
            if (UtilTimerStack.isActive())
                UtilTimerStack.pop(this.getClass().getName() + "_delegating_getGroup(attr)");
        }
    }

    private String getGroupDNFromMembershipAttribute(Attribute groupMembershipAtt) throws EntityException
    {
        if (UtilTimerStack.isActive())
            UtilTimerStack.push(this.getClass().getName() + "_delegating_getGroupDNFromMembershipAttribute");

        String groupDN = null;

        try
        {
            NamingEnumeration groupAttrs = groupMembershipAtt.getAll();

            while (groupAttrs.hasMoreElements())
            {
                groupDN = (String) groupAttrs.nextElement();
                if (groupDN != null)
                    break;
            }
        }
        catch (NamingException e)
        {
            throw new RepositoryException(e);
        }

        if (UtilTimerStack.isActive())
            UtilTimerStack.pop(this.getClass().getName() + "_delegating_getGroupDNFromMembershipAttribute");

        return groupDN;
    }

    /**
     * @param groupDN - String resembling a DN, e.g. "cn=groupA,ou=hr,ou=california,ou=utopia".
     * @return String holding the extracted groupName or the original String, if the argument does not resemble a DN
     */
    private String getGroupNameFromDN(String groupDN)
    {
        if (groupDN.indexOf("=") == -1)
            return groupDN;

        String[] names = groupDN.split(",");

        return names[0].split("=")[1];
    }

    public Entity getEntity(Attributes attrs, String distinguishedName) throws EntityException
    {
        return getGroup(attrs, distinguishedName);
    }

    public Entity getEntity(Attribute attr, String distinguishedName) throws EntityException
    {
        return getGroup(attr);
    }
}
