package com.atlassian.user.impl.hibernate;

import com.atlassian.user.Entity;
import com.atlassian.user.EntityException;
import com.atlassian.user.User;
import com.atlassian.user.UserManager;
import com.atlassian.user.impl.DuplicateEntityException;
import com.atlassian.user.impl.RepositoryException;
import com.atlassian.user.impl.hibernate.repository.HibernateRepository;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.page.DefaultPager;
import com.atlassian.user.search.page.Pager;
import com.atlassian.user.security.password.PasswordEncryptor;
import net.sf.hibernate.HibernateException;
import net.sf.hibernate.Query;
import net.sf.hibernate.Session;
import org.apache.log4j.Logger;
import org.springframework.dao.DataAccessException;
import org.springframework.orm.hibernate.HibernateCallback;
import org.springframework.orm.hibernate.SessionFactoryUtils;
import org.springframework.orm.hibernate.support.HibernateDaoSupport;

import java.util.Iterator;
import java.util.List;
import java.util.Set;

public class HibernateUserManager extends HibernateDaoSupport implements UserManager
{
    private static final String USERNAME_FIELD = "username";
    public static final String ENTITYID_FIELD = "entityid";

    protected final Logger log = Logger.getLogger(this.getClass());

    private final RepositoryIdentifier identifier;
    private final PasswordEncryptor passwordEncryptor;

    public HibernateUserManager(RepositoryIdentifier identifier, HibernateRepository repository, PasswordEncryptor passwordEncryptor)
    {
        this.identifier = identifier;
        this.passwordEncryptor = passwordEncryptor;
        setSessionFactory(repository.getSessionFactory());
    }

    public Pager getUsers() throws EntityException
    {
        List result;

        try
        {
            result = getHibernateTemplate().executeFind(new HibernateCallback()
            {
                public Object doInHibernate(Session session) throws HibernateException
                {
                    Query queryObject = session.getNamedQuery("atluser.user_findAll");
                    SessionFactoryUtils.applyTransactionTimeout(queryObject, getSessionFactory());

                    return queryObject.list();
                }
            });
        }
        catch (DataAccessException e)
        {
            throw new RepositoryException(e);
        }

        return new DefaultPager(result);
    }

    public Pager getUserNames() throws EntityException
    {
        List result;

        try
        {
            result = getHibernateTemplate().executeFind(new HibernateCallback()
            {
                public Object doInHibernate(Session session) throws HibernateException
                {
                    Query queryObject = session.getNamedQuery("atluser.user_findAllUserNames");
                    SessionFactoryUtils.applyTransactionTimeout(queryObject, getSessionFactory());

                    return queryObject.list();
                }
            });
        }
        catch (DataAccessException e)
        {
            throw new RepositoryException(e);
        }

        return new DefaultPager(result);
    }

    /**
     * @return an {@link User} if one could be found, otherwise null.
     * @throws EntityException representing the exception which prohibited looking for or retrieving the user.
     */
    public User getUser(final String username) throws EntityException
    {
        if (username == null)
            throw new IllegalArgumentException("Input (username) is null.");

        List result;

        try
        {
            result = getHibernateTemplate().executeFind(new HibernateCallback()
            {
                public Object doInHibernate(Session session) throws HibernateException
                {
                    Query queryObject = session.getNamedQuery("atluser.user_find");
                    SessionFactoryUtils.applyTransactionTimeout(queryObject, getSessionFactory());

                    if (username != null)
                        queryObject.setParameter(USERNAME_FIELD, username);

                    return queryObject.list();
                }
            });
        }
        catch (DataAccessException e)
        {
            throw new RepositoryException(e);
        }

        return result.isEmpty() ? null : (User) result.get(0);
    }

    public User createUser(String username) throws EntityException
    {
        if (username == null)
            throw new IllegalArgumentException("Input (username) is null.");

        User existingUser = getUser(username);
        if (existingUser != null)
            throw new DuplicateEntityException("User with name [" + username + "] already exists in this repository (" + identifier.getName() + ")");

        User user = new DefaultHibernateUser(username);
        saveUser(user);

        return user;
    }

    /**
     * Encrypts the plain password, sets it on the user, and saves the user.
     */
    public void alterPassword(User user, String password) throws EntityException
    {
        /**
         * I don't like this technique but it will work until I understand why I'm seeing:
         * "a different object with the same identifier value was already associated with the session" Hibernate Exceptions
         */
        User foundUser = getUser(user.getName());
        if (foundUser == null)
            throw new EntityException("This repository [" + identifier.getName() + "] does not handle user [" + user.getName() + "]");

        password = passwordEncryptor.encrypt(password);
        foundUser.setPassword(password);
        saveUser(foundUser);
    }

    public void saveUser(User user) throws EntityException
    {
        if (user == null)
            throw new IllegalArgumentException("Input (user) is null.");
        else if (!isHandledUser(user))
            throw new IllegalArgumentException("User is not a Hibernate entity [" + user.getClass().getName() + "]");

        DefaultHibernateUser dUser = (DefaultHibernateUser) user;

        try
        {
            getHibernateTemplate().saveOrUpdate(dUser);
            getSession().flush();
        }
        catch (HibernateException e)
        {
            throw new RepositoryException(e);
        }
    }

    /**
     * Removes the specified group, if it is present.
     *
     * @throws EntityException if an exception which prohibited removal
     */
    public void removeUser(User user) throws EntityException
    {
        if (user == null)
            throw new IllegalArgumentException("Input (username) is null.");
        else if (!isHandledUser(user))
            throw new IllegalArgumentException("User is not a Hibernate entity [" + user.getClass().getName());

        DefaultHibernateUser dUser = (DefaultHibernateUser) getUser(user.getName());
        if (dUser == null)
            throw new IllegalArgumentException("User can not be found in this user manager: [" + user + "]");

        List groups = getGroupsForLocalUser(dUser);

        if (groups != null)
        {
            dUser.setGroups(null);

            Iterator groupsOfUser = groups.iterator();

            while (groupsOfUser.hasNext())
            {
                DefaultHibernateGroup group = (DefaultHibernateGroup) groupsOfUser.next();
                Set members = group.getLocalMembers();

                if (members != null)
                    members.remove(dUser);

                getHibernateTemplate().saveOrUpdate(group);
            }
        }

        getHibernateTemplate().delete(dUser);
    }

    protected List getGroupsForLocalUser(final User user) throws RepositoryException
    {
        if (user == null)
            throw new IllegalArgumentException("Input (user) is null.");

        List result;
        final DefaultHibernateUser defUser = (DefaultHibernateUser) user;

        try
        {
            result = getHibernateTemplate().executeFind(new HibernateCallback()
            {
                public Object doInHibernate(Session session) throws HibernateException
                {
                    Query queryObject = session.getNamedQuery("atluser.group_getGroupsForUser");
                    SessionFactoryUtils.applyTransactionTimeout(queryObject, getSessionFactory());
                    queryObject.setLong(ENTITYID_FIELD, defUser.getId());

                    return queryObject.list();
                }
            });
        }
        catch (DataAccessException e)
        {
            throw new RepositoryException(e);
        }

        return result;
    }

    private boolean isHandledUser(User user)
    {
        return user instanceof DefaultHibernateUser;
    }

    public boolean isReadOnly(User user) throws EntityException
    {
        return false;
    }

    /**
     * @return a {@link com.atlassian.user.security.password.PasswordEncryptor} which handles the encrypytion of passwords for users managed by this
     *         object.
     * @throws UnsupportedOperationException - for {@link com.atlassian.user.UserManager} objects which do not create {@link com.atlassian.user.User} objects.
     */
    public PasswordEncryptor getPasswordEncryptor(User user) throws EntityException
    {
        return passwordEncryptor;
    }

    /**
     * @return the {@link com.atlassian.user.repository.RepositoryIdentifier} which is managed by this instance.
     */
    public RepositoryIdentifier getIdentifier()
    {
        return identifier;
    }

    public RepositoryIdentifier getRepository(Entity entity) throws EntityException
    {
        return getUser(entity.getName()) == null ? null : identifier;
    }

    /**
     * Used to detemine whether an entity can be added (eg, can call {@link com.atlassian.user.UserManager#createUser(String)} or {@link com.atlassian.user.GroupManager#createGroup(String)}
     */
    public boolean isCreative()
    {
        return true;
    }
}
