package com.atlassian.user.impl.hibernate;

import com.atlassian.user.*;
import com.atlassian.user.impl.DuplicateEntityException;
import com.atlassian.user.impl.RepositoryException;
import com.atlassian.user.impl.hibernate.repository.HibernateRepository;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.page.DefaultPager;
import com.atlassian.user.search.page.Pager;
import com.atlassian.user.search.page.PagerFactory;
import net.sf.hibernate.HibernateException;
import net.sf.hibernate.Query;
import net.sf.hibernate.Session;
import org.springframework.dao.DataAccessException;
import org.springframework.orm.hibernate.HibernateCallback;
import org.springframework.orm.hibernate.HibernateTemplate;
import org.springframework.orm.hibernate.SessionFactoryUtils;
import org.springframework.orm.hibernate.support.HibernateDaoSupport;

import java.util.*;

/**
 * A HibernateGroupManager which handles membership for local and external entities.
 */
public class HibernateGroupManager extends HibernateDaoSupport implements GroupManager
{
    public static final String GROUPNAME_FIELD = "groupname";
    public static final String GROUPID_FIELD = "groupid";
    public static final String ENTITYID_FIELD = "entityid";
    public static final String EXTERNAL_ENTITY_NAME_FIELD = "externalEntityName";

    private final RepositoryIdentifier identifier;
    protected final HibernateRepository repository;
    protected final UserManager userManager;
    protected final ExternalEntityDAO externalEntityDao;

    public HibernateGroupManager(RepositoryIdentifier identifier, HibernateRepository repository, UserManager userManager, ExternalEntityDAO externalEntityDao)
    {
        this.identifier = identifier;
        this.repository = repository;
        this.userManager = userManager;
        setSessionFactory(repository.getSessionFactory());
        this.externalEntityDao = externalEntityDao;
    }

    public Pager getGroups() throws EntityException
    {
        List result;

        try
        {
            result = getHibernateTemplate().executeFind(new HibernateCallback()
            {
                public Object doInHibernate(Session session) throws HibernateException
                {
                    Query queryObject = session.getNamedQuery("atluser.group_findAll");
                    SessionFactoryUtils.applyTransactionTimeout(queryObject, getSessionFactory());
                    return queryObject.list();
                }
            });
        }
        catch (DataAccessException e)
        {
            throw new RepositoryException(e);
        }

        if (result == null)
            return new DefaultPager();

        return new DefaultPager(result);
    }

    public Pager getGroups(User user) throws EntityException
    {
        Collection groups = getAllGroupsForUser(user);

        return new DefaultPager(groups);
    }

    protected Collection getAllGroupsForUser(User user) throws RepositoryException
    {
        if (user == null)
            throw new IllegalArgumentException("Input (user) is null.");

        Collection groups;

        if (isUserExternal(user))
            groups = getGroupsForExternalEntity(getCorrespondingExternalEntity(user));
        else
            groups = getGroupsForLocalUser(user);
        return groups;
    }

    /**
     * for the time being, lets define an internal user as a hibernate user. All other impl's of User are external users.
     */
    protected boolean isUserExternal(User user)
    {
        return !(user instanceof DefaultHibernateUser);
    }

    protected List getGroupsForLocalUser(final User user) throws RepositoryException
    {
        if (user == null)
            throw new IllegalArgumentException("Input (user) is null.");
        if (isUserExternal(user))
            return Collections.EMPTY_LIST;

        List result;
        final DefaultHibernateUser defUser = (DefaultHibernateUser) user;

        try
        {
            result = getHibernateTemplate().executeFind(new HibernateCallback()
            {
                public Object doInHibernate(Session session) throws HibernateException
                {
                    Query queryObject = session.getNamedQuery("atluser.group_getGroupsForUser");
                    SessionFactoryUtils.applyTransactionTimeout(queryObject, getSessionFactory());
                    queryObject.setLong(ENTITYID_FIELD, defUser.getId());

                    return queryObject.list();
                }
            });
        }
        catch (DataAccessException e)
        {
            throw new RepositoryException(e);
        }

        return result;
    }

    protected List getGroupsForExternalEntity(final ExternalEntity externalEntity) throws RepositoryException
    {
        if (externalEntity == null)
            throw new IllegalArgumentException("Input (externalEntity) is null.");

        List result;

        try
        {
            result = getHibernateTemplate().executeFind(new HibernateCallback()
            {
                public Object doInHibernate(Session session) throws HibernateException
                {
                    Query queryObject = session.getNamedQuery("atluser.group_getGroupsForExternalEntity");
                    SessionFactoryUtils.applyTransactionTimeout(queryObject, getSessionFactory());

                    if (externalEntity != null)
                        queryObject.setLong(ENTITYID_FIELD, externalEntity.getId());

                    return queryObject.list();
                }
            });
        }
        catch (DataAccessException e)
        {
            throw new RepositoryException(e);
        }

        return result;
    }

    /**
     * @return a {@link Pager} instance which can hold {@link User} <b>and</b> {@link ExternalEntity} objects.
     */
    public Pager getMemberNames(Group group) throws EntityException
    {
        if (group == null)
            throw new IllegalArgumentException("Group cannot be null.");
        if (!isHandledGroup(group))
            throw new IllegalArgumentException("Group passed to HibernateGroupManager must be of type 'DefaultHibernateGroup'");

        return PagerFactory.getPager(getExternalMemberNames(group), getLocalMemberNames(group));
    }

    protected void validateGroup(Group group)
    {
        if (group == null)
            throw new IllegalArgumentException("Input (group) is null.");
    }

    public Pager getLocalMemberNames(Group group) throws EntityException
    {
        validateGroup(group);

        final DefaultHibernateGroup defGroup = (DefaultHibernateGroup) group;
        List result;

        try
        {
            result = getHibernateTemplate().executeFind(new HibernateCallback()
            {
                public Object doInHibernate(Session session) throws HibernateException
                {
                    Query queryObject = session.getNamedQuery("atluser.group_getLocalMemberNames");
                    SessionFactoryUtils.applyTransactionTimeout(queryObject, getSessionFactory());

                    if (defGroup != null)
                        queryObject.setLong(GROUPID_FIELD, defGroup.getId());

                    return queryObject.list();
                }
            });
        }
        catch (DataAccessException e)
        {
            throw new RepositoryException(e);
        }

        return new DefaultPager(result);
    }


    public Pager getLocalMembers(Group group) throws RepositoryException
    {
        if (group == null)
            throw new IllegalArgumentException("Input (group) is null.");
        else if (!isHandledGroup(group)) //nothing here for a non-Hibernate group
            return new DefaultPager();

        final DefaultHibernateGroup defGroup = (DefaultHibernateGroup) group;
        List result = new ArrayList(defGroup.getLocalMembers());

        return new DefaultPager(result);
    }

    private boolean isHandledGroup(Group group)
    {
        return (group instanceof DefaultHibernateGroup);
    }

    public Pager getExternalMembers(Group group) throws RepositoryException
    {
        if (group == null)
            throw new IllegalArgumentException("Input (group) is null.");
        else if (!isHandledGroup(group))
            return new DefaultPager();

        final DefaultHibernateGroup defGroup = (DefaultHibernateGroup) group;
        List result = new ArrayList(defGroup.getExternalMembers());

        return new DefaultPager(result);
    }

    public Pager getExternalMemberNames(Group group) throws EntityException
    {
        if (group == null)
            throw new IllegalArgumentException("Input (group) is null.");
        else if (!isHandledGroup(group))
            return new DefaultPager();

        final DefaultHibernateGroup defGroup = (DefaultHibernateGroup) group;
        List result;

        try
        {
            result = getHibernateTemplate().executeFind(new HibernateCallback()
            {
                public Object doInHibernate(Session session) throws HibernateException
                {
                    Query queryObject = session.getNamedQuery("atluser.group_getExternalMemberNames");
                    SessionFactoryUtils.applyTransactionTimeout(queryObject, getSessionFactory());

                    if (defGroup != null)
                        queryObject.setLong(GROUPID_FIELD, defGroup.getId());

                    return queryObject.list();
                }
            });
        }
        catch (DataAccessException e)
        {
            throw new RepositoryException(e);
        }

        return new DefaultPager(result);
    }

    public Group getGroup(final String groupname) throws EntityException
    {
        if (groupname == null)
            throw new IllegalArgumentException("Input (groupname) is null.");

        List result;
        Group foundGroup = null;

        try
        {
            result = getHibernateTemplate().executeFind(new HibernateCallback()
            {
                public Object doInHibernate(Session session) throws HibernateException
                {
                    Query queryObject = session.getNamedQuery("atluser.group_find");
                    SessionFactoryUtils.applyTransactionTimeout(queryObject, getSessionFactory());

                    if (groupname != null)
                        queryObject.setParameter(GROUPNAME_FIELD, groupname);

                    return queryObject.list();
                }
            });
        }
        catch (DataAccessException e)
        {
            throw new RepositoryException(e);
        }

        try
        {
            foundGroup = (Group) result.get(0);
        }
        catch (Exception e)
        {
            return foundGroup;
        }

        return foundGroup;
    }

    public Group createGroup(String groupname) throws EntityException
    {
        if (groupname == null)
            throw new IllegalArgumentException("Input (groupname) is null.");

        Group group = getGroup(groupname);

        if (group == null)
        {
            group = new DefaultHibernateGroup(groupname);
            getHibernateTemplate().save(group);
        }
        else
            throw new DuplicateEntityException("Group [" + groupname + "] already exists in this repository (" +
                    identifier.getName() + ")");

        return group;
    }

    /**
     * Removes the specified group, if it is present.
     *
     * @throws com.atlassian.user.EntityException
     *          - representing the exception which prohibited removal
     */
    public void removeGroup(Group group) throws EntityException
    {
        Group groupInSession = getGroupInSession(group);

        DefaultHibernateGroup dGroup = (DefaultHibernateGroup) groupInSession;
        dGroup.setExternalMembers(null);
        dGroup.setLocalMembers(null);

        getHibernateTemplate().delete(groupInSession);
    }

    public void addMembership(Group group, User user) throws EntityException
    {
        validateGroupAndUser(group, user);

        Group groupInSession = getGroupInSession(group);

        DefaultHibernateGroup dGroup = (DefaultHibernateGroup) groupInSession;
        Set membership;
        Entity member;

        if (isUserExternal(user))
        {
            membership = dGroup.getExternalMembers();

            if (membership == null)
            {
                membership = new HashSet();
                dGroup.setExternalMembers(membership);
            }

            member = getCorrespondingExternalEntity(user);
            membership.add(member);
        }
        else
        {
            membership = dGroup.getLocalMembers();

            if (membership == null)
            {
                membership = new HashSet();
                dGroup.setLocalMembers(membership);
            }

            membership.add(user);
        }



        try
        {
            getHibernateTemplate().saveOrUpdate(dGroup);
            getSession().flush();
        }
        catch (HibernateException e)
        {
            throw new EntityException(e);
        }
    }

    protected ExternalEntity getCorrespondingExternalEntity(final User user) throws RepositoryException
    {
        if (user == null)
            throw new IllegalArgumentException("Input (user) is null.");

        ExternalEntity result = externalEntityDao.getExternalEntity(user.getName());

        if (result == null)
            return createExternalEntity(user);
        else
            return result;
    }

    protected ExternalEntity createExternalEntity(User user)
    {
        DefaultHibernateExternalEntity entity = new DefaultHibernateExternalEntity();
        entity.setName(user.getName());
        entity.setType(user.getClass().getName());
        externalEntityDao.saveExternalEntity(entity);
        return entity;
    }

    public boolean hasMembership(Group group, User user) throws EntityException
    {
        if (group==null || getGroup(group.getName()) == null)
            return false;

        validateGroupAndUser(group, user);

        final DefaultHibernateGroup defGroup = (DefaultHibernateGroup) getGroupInSession(group);

        if (isUserExternal(user))
        {
            return hasExternalMembership(defGroup, user);
        }
        else
        {
            return hasLocalMembership(defGroup, (DefaultHibernateUser) user);
        }
    }

    protected void validateGroupAndUser(Group group, User user) throws EntityException
    {
        if (group == null)
            throw new IllegalArgumentException("Can't add membership for null group");

        if (getGroup(group.getName()) == null)
            throw new IllegalArgumentException("Group unknown: [" + group + "] in [" + identifier.getKey() + "]");

        if (user == null)
            throw new IllegalArgumentException("User unknown: [" + user + "] in [" + identifier.getKey() + "]");

        if (!isHandledGroup(group))
            throw new IllegalArgumentException("Group is not a Hibernate entity [" + group.getClass().getName());
    }

    protected boolean hasExternalMembership(DefaultHibernateGroup defGroup, User user) throws EntityException
    {
        ExternalEntity entity = getCorrespondingExternalEntity(user);

        return defGroup.getExternalMembers().contains(entity);
    }

    protected boolean hasLocalMembership(DefaultHibernateGroup defGroup, DefaultHibernateUser defUser) throws EntityException
    {
        Collection usersGroups = getAllGroupsForUser(defUser);

        return (usersGroups != null && usersGroups.contains(defGroup));
    }

    public void removeMembership(Group group, User user) throws EntityException
    {
        validateGroupAndUser(group, user);

        Group groupInSession = getGroupInSession(group);

        DefaultHibernateGroup hibernateGroup = (DefaultHibernateGroup) groupInSession;

        HibernateTemplate hibernateTemplate = getHibernateTemplate();
        if (isUserExternal(user))
        {
            ExternalEntity extUser = getCorrespondingExternalEntity(user);

            // remove user from group
            hibernateGroup.getExternalMembers().remove(extUser);

        }
        else
        {
            // remove user from group
            hibernateGroup.getLocalMembers().remove(user);

        }

        hibernateTemplate.saveOrUpdate(groupInSession);
        hibernateTemplate.flush();
    }

    public boolean isReadOnly(Group group) throws EntityException
    {
        return (getGroup(group.getName()) == null);
    }

    public boolean supportsExternalMembership() throws EntityException
    {
        return false;
    }


    /**
     * @return the {@link com.atlassian.user.repository.RepositoryIdentifier} which is managed by this instance.
     */
    public RepositoryIdentifier getIdentifier()
    {
        return identifier;
    }

    public RepositoryIdentifier getRepository(Entity entity) throws EntityException
    {
        if (getGroup(entity.getName()) != null)
            return identifier;

        return null;
    }

    /**
     * Used to detemine whether an entity can be added (eg, can call {@link com.atlassian.user.UserManager#createUser(String)} or
     * {@link com.atlassian.user.GroupManager#createGroup(String)}
     */
    public boolean isCreative()
    {
        return true;
    }

    /**
     * Used to get a group from Hibernate (in session) from the "cached" version.
     * Ensures that we do not have un-attached group objects.
     */
    private Group getGroupInSession(Group group)
            throws EntityException
    {
        if (group == null)
            throw new IllegalArgumentException("Input (group) is null.");
        else if (!isHandledGroup(group))
            throw new IllegalArgumentException("Group is not a Hibernate entity [" + group.getClass().getName());

        return getGroup(group.getName());
    }
}