package com.atlassian.user.search.query;

/**
 * This {@link EntityQueryParser} is not efficient, as it handles all queries by loading
 * user and group {@link Entity} objects into memory.
 */

import com.atlassian.user.*;
import com.atlassian.user.configuration.Configuration;
import com.atlassian.user.configuration.ConfigurationException;
import com.atlassian.user.configuration.util.InitializationCheck;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.DefaultSearchResult;
import com.atlassian.user.search.SearchResult;
import com.atlassian.user.search.query.match.*;
import com.atlassian.user.search.page.DefaultPager;
import com.atlassian.user.search.page.Pager;
import com.atlassian.user.search.page.PagerUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.*;

public class DefaultEntityQueryParser extends AbstractEntityQueryParser implements EntityQueryParser
{
    private static final Logger log = Logger.getLogger(DefaultEntityQueryParser.class);

    protected UserManager userManager;
    protected GroupManager groupManager;
    protected RepositoryIdentifier repository;
    protected Method entityNameMethod;
    protected Method emailMethod;
    protected Method fullnameMethod;
    private static final Class userClass = User.class;

    public DefaultEntityQueryParser(RepositoryIdentifier repo, UserManager userManager, GroupManager groupManager)
    {
        try
        {
            entityNameMethod = userClass.getMethod("getName", null);
            emailMethod = userClass.getMethod("getEmail", null);
            fullnameMethod = userClass.getMethod("getFullName", null);
        }
        catch (NoSuchMethodException e)
        {
            log.error(e.getMessage());
        }

        this.userManager = userManager;
        this.groupManager = groupManager;
        this.repository = repo;
    }

    public void init(HashMap args) throws ConfigurationException
    {
        this.userManager = (UserManager) args.get(Configuration.USERMANAGER);
        this.groupManager = (GroupManager) args.get(Configuration.GROUPMANAGER);
        this.repository = (RepositoryIdentifier) args.get(Configuration.REPOSITORY);

        InitializationCheck.validateArgs(args, new String[]{Configuration.USERMANAGER,
            Configuration.GROUPMANAGER,
            Configuration.REPOSITORY}, this);

        try
        {
            entityNameMethod = userClass.getMethod("getName", null);
            emailMethod = userClass.getMethod("getEmail", null);
            fullnameMethod = userClass.getMethod("getFullName", null);
        }
        catch (NoSuchMethodException e)
        {
            log.error(e.getMessage());
        }
    }

    protected Pager parseQuery(Method userMethod, TermQuery q, Pager data)
        throws IllegalAccessException, InvocationTargetException
    {
        String searchTerm = StringUtils.defaultString(q.getTerm()).toLowerCase(); // to allow case insensitive search
        if (searchTerm.indexOf(TermQuery.WILDCARD) >= 0)
            return data;

        Matcher matcher;
        if (q.isMatchingSubstring())
        {
            if (q.getMatchingRule().equals(TermQuery.SUBSTRING_STARTS_WITH))
            {
                matcher = new StartsWithIgnoreCaseMatcher();
            }
            else if (q.getMatchingRule().equals(TermQuery.SUBSTRING_ENDS_WITH))
            {
                matcher = new EndsWithIgnoreCaseMatcher();
            }
            else
            {
                matcher = new ContainsIgnoreCaseMatcher();
            }
        }
        else
        {
            matcher = new EqualsIgnoreCaseMatcher();
        }

        List matches = new ArrayList();
        for (Iterator iterator = data.iterator(); iterator.hasNext();)
        {
            Entity entity = (Entity) iterator.next();
            String userInfo = (String) userMethod.invoke(entity, null);

            if (matcher.matches(userInfo, searchTerm))
                matches.add(entity);
        }
        return new DefaultPager(matches);
    }

    public Pager find(Query query) throws EntityException
    {
        Pager result = null;

        if (query instanceof TermQuery)
        {
            try
            {
                if (query instanceof UserNameTermQuery)
                    result = parseQuery(entityNameMethod, (TermQuery) query, userManager.getUsers());
                else if (query instanceof GroupNameTermQuery)
                    result = parseQuery(entityNameMethod, (TermQuery) query, groupManager.getGroups());
                else if (query instanceof EmailTermQuery)
                    result = parseQuery(emailMethod, (TermQuery) query, userManager.getUsers());
                else if (query instanceof FullNameTermQuery)
                    result = parseQuery(fullnameMethod, (TermQuery) query, userManager.getUsers());
            }
            catch (IllegalAccessException e)
            {
                throw new EntityException(e);
            }
            catch (InvocationTargetException e)
            {
                throw new EntityException(e);
            }
        }
        else if (query instanceof MembershipQuery)
        {
            result = parseMembershipQuery(query);
        }
        else if (query instanceof BooleanQuery)
        {
            BooleanQuery bQuery = (BooleanQuery) query;
            result = evaluateBoolean(bQuery);
        }

        return result;
    }

    private Pager parseMembershipQuery(Query query) throws EntityException
    {
        Pager result = null;

        if (query instanceof UsersInGroupTwoTermQuery)
        {
            result = parseUsersInGroupTwoTermQuery((UsersInGroupTwoTermQuery) query);
        }
        else if (query instanceof GroupsOfUserTwoTermQuery)
        {
            result = parseGroupsOfUserTwoTermQuery((GroupsOfUserTwoTermQuery) query);
        }

//        TermQuery tQuery = (TermQuery) query;
//
//        if (tQuery.isMatchingSubstring())
//        {
//            GroupNameTermQuery q = new GroupNameTermQuery(tQuery.getTerm(), tQuery.getMatchingRule());
//            Pager pager = find(q);
//            Iterator iter = pager.iterator();
//
//            if (pager.isEmpty())
//                return new DefaultPager();
//            else
//            {
//                Pager members = null;
//                while (iter.hasNext())
//                {
//                    Group group = (Group) iter.next();
//
//                    if (members == null)
//                        members = groupManager.getMemberNames(group);
//                    else
//                        members = new MergedPager(members, groupManager.getMemberNames(group));
//                }
//
//                return members;
//            }
//
//        }
//        else
//            return groupManager.getMemberNames(groupManager.getGroup(tQuery.getTerm()));

        return result;
    }

    private Pager parseUsersInGroupTwoTermQuery(UsersInGroupTwoTermQuery query) throws EntityException
    {
        UserNameTermQuery userQuery = query.getUserNameTermQuery();
        GroupNameTermQuery groupQuery = query.getGroupNameTermQuery();

        Pager users = find(userQuery);
        Pager groups = find(groupQuery);

        Iterator groupsIter = groups.iterator();
        Set candidateUsers = new HashSet();

        while (groupsIter.hasNext())
        {
            Group group = (Group) groupsIter.next();
            candidateUsers.addAll(PagerUtils.toList(groupManager.getLocalMemberNames(group)));
        }

        List userQueryList = PagerUtils.toList(users);

        userQueryList.retainAll(candidateUsers);

        return new DefaultPager(userQueryList);
    }

    private Pager evaluateBoolean(BooleanQuery query) throws EntityException
    {
        List queries = query.getQueries();
        Pager allResults = null;

        if (query instanceof MembershipQuery)
        {
            return parseMembershipQuery(query);
        }

        boolean anding = query.isAND();

        for (int i = 0; i < queries.size(); i++)
        {
            Query nextQuery = (Query) queries.get(i);
            List initialResult;

            try
            {
                if (allResults == null)
                {
                    allResults = find(nextQuery);
                }
                else if (nextQuery instanceof BooleanQuery)
                {
                    if (anding)
                    {
                        Pager resultsToAnd = evaluateBoolean((BooleanQuery) nextQuery);
                        List resultsToAndList = PagerUtils.toList(resultsToAnd);
                        List allResultsList = PagerUtils.toList(allResults);

                        resultsToAndList.retainAll(allResultsList);
                        allResults = new DefaultPager(resultsToAndList);
                    }
                    else
                    {
                        Pager resultsToOr = evaluateBoolean((BooleanQuery) nextQuery);
                        List resultsToOrList = PagerUtils.toList(resultsToOr);
                        List intersection = findIntersection(PagerUtils.toList(allResults), resultsToOrList);
                        allResults = new DefaultPager(intersection);
                    }
                }
                else if (anding)
                {

                    if (nextQuery instanceof UserNameTermQuery)
                    {
                        initialResult = PagerUtils.toList(parseQuery(entityNameMethod, (TermQuery) nextQuery,
                            allResults));
                        initialResult.addAll(PagerUtils.toList(allResults));
                        allResults = new DefaultPager(initialResult);
                    }
                    else if (nextQuery instanceof GroupNameTermQuery)
                    {
                        initialResult = PagerUtils.toList(parseQuery(entityNameMethod, (TermQuery) nextQuery,
                            allResults));
                        initialResult.addAll(PagerUtils.toList(allResults));
                        allResults = new DefaultPager(initialResult);
                    }
                    else if (nextQuery instanceof EmailTermQuery)
                    {
                        initialResult = PagerUtils.toList(parseQuery(emailMethod, (TermQuery) nextQuery, allResults));
                        initialResult.addAll(PagerUtils.toList(allResults));
                        allResults = new DefaultPager(initialResult);
                    }
                    else if (nextQuery instanceof FullNameTermQuery)
                    {
                        initialResult = PagerUtils.toList(parseQuery(fullnameMethod, (TermQuery) nextQuery,
                            allResults));
                        initialResult.addAll(PagerUtils.toList(allResults));
                        allResults = new DefaultPager(initialResult);
                    }
                }
                else // OR
                {
                    initialResult = PagerUtils.toList(find(nextQuery));
                    List intersection = findIntersection(PagerUtils.toList(allResults), initialResult);
                    allResults = new DefaultPager(intersection);
                }
            }
            catch (Exception e)
            {
                log.error(e.getClass().getName() + " - " + e.getMessage());
            }
        }

        return allResults;
    }

    private List findIntersection(List list1, List list2)
    {
        list2.removeAll(list1);
        list1.addAll(list2);

        return list1;
    }

    private Pager parseGroupsOfUserTwoTermQuery(GroupsOfUserTwoTermQuery query) throws EntityException
    {
        UserNameTermQuery userQuery = query.getUserNameTermQuery();
        GroupNameTermQuery groupQuery = query.getGroupNameTermQuery();

        Pager groups = find(groupQuery);
        Pager users = find(userQuery);

        Iterator usersIter = users.iterator();
        Set candidateGroups = new HashSet();

        while (usersIter.hasNext())
        {
            User user = (User) usersIter.next();
            candidateGroups.addAll(PagerUtils.toList(groupManager.getGroups(user)));
        }

        List groupQueryList = PagerUtils.toList(groups);

        groupQueryList.retainAll(candidateGroups);

        return new DefaultPager(groupQueryList);
    }

    public SearchResult findUsers(Query query) throws EntityException
    {
        validateQuery(query);
        Pager pager = find(query);
        return new DefaultSearchResult(pager, repository.getKey());
    }

    public SearchResult findGroups(Query query) throws EntityException
    {
        validateQuery(query);
        Pager pager = find(query);
        return new DefaultSearchResult(pager, repository.getKey());
    }

    public SearchResult findUsers(Query query, QueryContext context) throws EntityException
    {
        validateQuery(query);
        SearchResult result = null;
        if (context.getRepositoryKeys().contains(repository.getKey()) || context.getRepositoryKeys().contains(
            QueryContext.ALL_REPOSITORIES))
        {
            result = findUsers(query);
        }

        return result;
    }

    public SearchResult findGroups(Query query, QueryContext context) throws EntityException
    {
        validateQuery(query);
        SearchResult result = null;

        if (context.getRepositoryKeys().contains(repository.getKey()) || context.getRepositoryKeys().contains(
            QueryContext.ALL_REPOSITORIES))
        {
            result = findGroups(query);
        }

        return result;
    }
}