package com.atlassian.user.search.query;

import java.util.List;
import java.util.Iterator;

public abstract class AbstractEntityQueryParser implements EntityQueryParser
{
    protected void validateQuery(Query query) throws EntityQueryException
    {
        if (query instanceof MembershipQuery)
            throw new IllegalArgumentException("Membership queries are illegal until we verify they can be performed on" +
                    " all LDAP system (i.e. the membership attribute must support substring matching");

        if (query == null)
            throw new IllegalArgumentException("Null arg.");

        if (!(query instanceof BooleanQuery))
            return;

        validateBooleanToQueryType((BooleanQuery) query, null);
    }

    /**
     * Ensures that the {@link BooleanQuery} combines queries of a singular type {@link UserQuery}, {@link GroupQuery}, or
     * {@link MembershipQuery}.
     */
    protected void validateBooleanToQueryType(BooleanQuery query, String queryClassName) throws EntityQueryException
    {
        List queries = query.getQueries();
        Iterator queryIter = queries.iterator();

        while (queryIter.hasNext())
        {
            Query foundQuery = (Query) queryIter.next();

            if (foundQuery instanceof BooleanQuery)
            {
                validateBooleanToQueryType((BooleanQuery) foundQuery, queryClassName);
                return;
            }

            if (queryClassName == null) //identify the type of query which the BooleanQuery is restricted to
            {
                queryClassName = matchQueryToQueryType(foundQuery);
            }
        }
    }

    protected String matchQueryToQueryType(Query query1)
    {
        String queryClassName = null;

        if (query1 instanceof UserQuery)
            queryClassName = UserQuery.class.getName();
        else if (query1 instanceof GroupQuery)
            queryClassName = GroupQuery.class.getName();
        else if (query1 instanceof MembershipQuery)
            queryClassName = MembershipQuery.class.getName();

        return queryClassName;
    }

    /**
     * The 'defining query' lets us know which EntityClass we are querying.
     */
    protected Query identifyDefiningQuery(BooleanQuery q)
    {
        if (q instanceof MembershipQuery)
            return q;

        Iterator iter = q.getQueries().iterator();
        TermQuery foundQuery = null;

        while (iter.hasNext())
        {
            Query query = (Query) iter.next();
            if (query instanceof TermQuery)
            {
                foundQuery = (TermQuery) query;
                break;
            }
            else if (query instanceof BooleanQuery)
                return identifyDefiningQuery((BooleanQuery) query);
        }

        return foundQuery;
    }
}
