package com.atlassian.user.impl.memory;

import com.atlassian.user.*;
import com.atlassian.user.impl.DefaultGroup;
import com.atlassian.user.impl.DuplicateEntityException;
import com.atlassian.user.impl.memory.provider.MemoryProvider;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.page.Pager;

public class MemoryGroupManager implements GroupManager
{
    private final MemoryProvider provider;
    private final RepositoryIdentifier repository;

    public MemoryGroupManager(RepositoryIdentifier repository, MemoryProvider provider)
    {
        this.provider = provider;
        this.repository = repository;
    }

    public Group createGroup(String groupName) throws EntityException
    {
        if (getGroup(groupName) != null)
            throw new DuplicateEntityException("The group [" + groupName + "] already exists.");

        Group group = new DefaultGroup(groupName);
        provider.addGroup(group);

        return group;
    }

    public void removeGroup(Group group) throws EntityException
    {
        provider.removeGroup(group);
    }

    public void addMembership(Group group, User user)
    {
        if (group == null)
            throw new IllegalArgumentException("Can't add membership for null group");
        if (getGroup(group.getName()) == null)
            throw new IllegalArgumentException("Can't add membership for unknown group: [" + group.getName() + "]");
        provider.addMembership(group, user);
    }

    public boolean hasMembership(Group group, User user)
    {
        return provider.hasMembership(group, user);
    }

    public void saveGroup(Group group) throws EntityException
    {
    }

    public Pager getGroups() throws EntityException
    {
        return provider.getGroups();

    }

    public Group getGroup(String groupName)
    {
        return provider.getGroup(groupName);
    }

    public Pager getGroups(User user) throws EntityException
    {
        return provider.getGroups(user);
    }

    public RepositoryIdentifier getConfiguration()
    {
        return repository;
    }

    /**
     * @return the {@link com.atlassian.user.repository.RepositoryIdentifier} which is managed by this instance.
     */
    public RepositoryIdentifier getIdentifier()
    {
        return repository;
    }

    public RepositoryIdentifier getRepository(Entity entity) throws EntityException
    {
        if (getGroup(entity.getName()) != null)
            return repository;

        return null;
    }

    public String getApplicationKey()
    {
        return null;
    }

    public boolean isCreative()
    {
        return true;
    }

    public void removeMembership(Group group, User user)
    {
        if (group==null || getGroup(group.getName()) == null)
            throw new IllegalArgumentException("Can't remove membership for unknown group: [" +
                (group==null ? "null" : group.getName()) + "]");
        provider.removeMembership(group, user);
    }

    public Pager getMemberNames(Group group)
    {
        return provider.getMemberNames(group);
    }

    public Pager getLocalMemberNames(Group group) throws EntityException
    {
        return provider.getLocalMembers(group);
    }

    public Pager getExternalMemberNames(Group group) throws EntityException
    {
        return provider.getExternalMembers(group);
    }

    public boolean isReadOnly(Group group) throws EntityException
    {
        return false;
    }

    public boolean supportsExternalMembership() throws EntityException
    {
        return true;
    }
}
