package com.atlassian.user.impl.ldap.search.page;

import com.atlassian.user.EntityException;
import com.atlassian.user.impl.ldap.repository.LdapContextFactory;
import com.atlassian.user.impl.ldap.search.LDAPPagerInfo;
import com.atlassian.user.search.page.AbstractPrefetchingPager;
import com.atlassian.user.util.EnumerationAdaptor;
import com.atlassian.user.util.LDAPUtils;
import com.atlassian.util.profiling.UtilTimerStack;
import net.sf.ldaptemplate.support.filter.Filter;
import org.apache.log4j.Logger;

import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.DirContext;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import java.util.Iterator;
import java.util.List;

public abstract class AbstractLDAPPager extends AbstractPrefetchingPager implements Iterator
{
    protected final Logger log = Logger.getLogger(this.getClass());
    protected NamingEnumeration enume;
    protected Filter originalQuery;
    protected LdapContextFactory repository;
    protected String originalBaseSearchContext;
    protected boolean searchAllDepths;
    protected String[] returningAttributes;
    private int timeLimitMillis;
    public boolean closed;

    protected AbstractLDAPPager(LdapContextFactory repository, LDAPPagerInfo info)
    {
        this.repository = repository;
        this.enume = info.getNamingEnumeration();
        this.originalQuery = info.getLDAPQuery();
        this.originalBaseSearchContext = info.getBaseSearchContext();
        this.searchAllDepths = info.isSearchAllDepths();
        this.returningAttributes = info.getReturningAttributes();
        this.timeLimitMillis = info.getTimeToLive();
    }

    public AbstractLDAPPager()
    {

    }

    /**
     * Parses the {@link SearchResult} and stores the result in the list of prefetched items.
     *
     * @return the {@link List} of prefetched results.
     */
    protected abstract List preloadSearchResult(SearchResult result, List prefetched) throws EntityException;

    public void remove()
    {
        throw new UnsupportedOperationException("This iterator does not support removal.");
    }

    /**
     * Preloads the page into the prefetched list.
     */
    protected void preload()
    {
        indexOfFirstItemInCurrentPage = idx;

        if (UtilTimerStack.isActive())
        {
            UtilTimerStack.push(this.getClass().getName() + "_preload__(originalQuery= " + originalQuery + ")");
        }

        DirContext ctx = null;

        try
        {
            if (closed)
            {
                ctx = repository.getLDAPContext();
                SearchControls ctls = LDAPUtils.createSearchControls(returningAttributes, searchAllDepths,
                    timeLimitMillis);
                if (log.isDebugEnabled())
                {
                    log.debug("AbstractLDAPPager.preload:" + originalQuery.encode());
                }
                enume = ctx.search(originalBaseSearchContext, originalQuery.encode(), ctls);
            }

            super.preload(new EnumerationAdaptor(enume));
        }
        catch (Exception e)
        {
            log.error("Could not close connection", e);
        }
        finally
        {
            if (enume != null)
            {
                try { enume.close(); } catch (NamingException e) { log.error("Error closing enumeration.", e); }
            }

            if (ctx != null)
            {
                try { ctx.close(); } catch (NamingException e) { log.error("Error closing context.", e); }
            }

            closed = true;
        }

        if (UtilTimerStack.isActive())
        {
            UtilTimerStack.pop(this.getClass().getName() + "_preload__(originalQuery= " + originalQuery + ")");
        }
    }

    protected List fetch(Object element, List prefetched) throws EntityException
    {
        return preloadSearchResult((SearchResult) element, prefetched);
    }

}