package com.atlassian.user.impl.ldap.repository;

import com.atlassian.user.impl.RepositoryException;
import com.atlassian.user.impl.ldap.properties.LdapConnectionProperties;
import com.atlassian.util.profiling.UtilTimerStack;

import javax.naming.NamingException;
import javax.naming.Context;
import javax.naming.directory.DirContext;
import javax.naming.directory.InitialDirContext;
import java.util.Hashtable;
import java.util.Map;
import java.util.HashMap;
import java.util.Iterator;

public class DefaultLdapContextFactory implements LdapContextFactory
{
    private final LdapConnectionProperties connectionProperties;

    public DefaultLdapContextFactory(LdapConnectionProperties connectionProperties)
    {
        this.connectionProperties = connectionProperties;
    }

    public Hashtable getJNDIEnv()
    {
        Map result = new HashMap();

        result.put(Context.PROVIDER_URL, connectionProperties.getProviderURL());
        result.put(Context.INITIAL_CONTEXT_FACTORY, connectionProperties.getJNDIInitialContextFactoryIdentifier());
        result.put(Context.SECURITY_AUTHENTICATION, connectionProperties.getSecurityAuthentication());
        result.put(Context.SECURITY_PRINCIPAL, connectionProperties.getSecurityPrincipal());
        result.put(Context.SECURITY_CREDENTIALS, connectionProperties.getSecurityCredential());
        result.put(Context.SECURITY_PROTOCOL, connectionProperties.getSecurityProtocol());
        result.put(Context.BATCHSIZE, Integer.toString(connectionProperties.getSearchBatchSize()));

        if (connectionProperties.isPoolingOn())
            result.put("com.sun.jndi.ldap.connect.pool", "true");

        String referral = System.getProperty("atlassian." + Context.REFERRAL);
        if (referral != null)
            result.put(Context.REFERRAL, referral);

        return createHashtable(result);
    }

    public Hashtable getAuthenticationJndiEnvironment(String userDn, String password)
    {
        Map env = new HashMap(getJNDIEnv());

        env.put(Context.SECURITY_PRINCIPAL, userDn);
        env.put(Context.SECURITY_CREDENTIALS, password);

        // context used for authentication must have "simple" authentication so that passed in
        // credentials are _never_ ignored (USER-139)
        env.put(Context.SECURITY_AUTHENTICATION, "simple");

        // don't use connection pooling on authentication connections (USER-137)
        env.put("com.sun.jndi.ldap.connect.pool", "false");
        return createHashtable(env);
    }

    /**
     * {@link Hashtable}s can't contain <tt>null</tt> as a value or key, so we check each entry in our map before adding
     * it to the Hashtable.
     */
    private Hashtable createHashtable(Map map)
    {
        Hashtable result = new Hashtable();
        for (Iterator entryIterator = map.entrySet().iterator(); entryIterator.hasNext();)
        {
            Map.Entry entry = (Map.Entry) entryIterator.next();
            if (entry.getValue() == null || entry.getKey() == null)
                continue;
            result.put(entry.getKey(), entry.getValue());
        }
        return result;
    }

    public DirContext getLDAPContext() throws RepositoryException
    {
        if (UtilTimerStack.isActive())
            UtilTimerStack.push(this.getClass().getName() + "_getLDAPContext");

        try
        {
            return new InitialDirContext(getJNDIEnv());
        }
        catch (NamingException e)
        {
            throw new RepositoryException(e);
        }
        finally
        {
            if (UtilTimerStack.isActive())
                UtilTimerStack.pop(this.getClass().getName() + "_getLDAPContext");
        }
    }
}
