package com.atlassian.user.impl.ldap.adaptor;

import com.atlassian.user.EntityException;
import com.atlassian.user.Group;
import com.atlassian.user.User;
import com.atlassian.user.impl.RepositoryException;
import com.atlassian.user.impl.ldap.LDAPEntity;
import com.atlassian.user.impl.ldap.LDAPGroupFactory;
import com.atlassian.user.impl.ldap.properties.LdapSearchProperties;
import com.atlassian.user.impl.ldap.properties.LdapMembershipProperties;
import com.atlassian.user.impl.ldap.repository.LdapContextFactory;
import com.atlassian.user.impl.ldap.search.DefaultLDAPUserAdaptor;
import com.atlassian.user.impl.ldap.search.LDAPPagerInfo;
import com.atlassian.user.impl.ldap.search.LDAPUserAdaptor;
import com.atlassian.user.impl.ldap.search.LdapFilterFactory;
import com.atlassian.user.impl.ldap.search.page.LDAPEntityPager;
import com.atlassian.user.impl.ldap.search.page.LDAPMembershipToUsernamePager;
import com.atlassian.user.search.page.DefaultPager;
import com.atlassian.user.search.page.Pager;
import com.atlassian.user.util.LDAPUtils;
import com.atlassian.util.profiling.UtilTimerStack;
import net.sf.ldaptemplate.support.filter.EqualsFilter;
import net.sf.ldaptemplate.support.filter.Filter;

public class LDAPStaticGroupAdaptor extends AbstractLDAPGroupAdaptor
{
    private final LDAPUserAdaptor userAdaptor;
    private final LdapMembershipProperties membershipProperties;

    public LDAPStaticGroupAdaptor(LdapContextFactory repository, LdapSearchProperties searchProperties,
        LDAPGroupFactory groupFactory, LdapFilterFactory filterFactory, LdapMembershipProperties membershipProperties)
    {
        super(repository, searchProperties, groupFactory, filterFactory);
        this.membershipProperties = membershipProperties;
        userAdaptor = new DefaultLDAPUserAdaptor(repository, searchProperties, filterFactory);
    }

    public Pager getGroups(User user) throws EntityException
    {
        LDAPPagerInfo info = getGroupEntriesViaMembership(user);

        if (info == null)
            return new DefaultPager();

        return new LDAPEntityPager(repository, groupFactory, info);
    }

    public Pager findMembers(Group group) throws EntityException
    {
        LDAPPagerInfo info =
            getGroupEntries(group.getName(), new String[]{membershipProperties.getMembershipAttribute()}, null);

        return new LDAPMembershipToUsernamePager(repository, info);
    }

    public Pager findMemberNames(Group group) throws EntityException
    {
        LDAPPagerInfo info =
            getGroupEntries(group.getName(), new String[]{membershipProperties.getMembershipAttribute()}, null);

        return new LDAPMembershipToUsernamePager(repository, info);
    }

    public boolean hasStaticGroups()
    {
        return true;
    }

    public boolean hasMembership(Group group, User user) throws RepositoryException
    {
        if (!(user instanceof LDAPEntity))
            return false;

        if (UtilTimerStack.isActive())
            UtilTimerStack.push(this.getClass().getName() + "_hasMembership(" + group + ", " + user + ")");

        Filter searchFilter = LDAPUtils.makeAndFilter(
            new EqualsFilter(searchProperties.getGroupnameAttribute(), group.getName()),
            new EqualsFilter(membershipProperties.getMembershipAttribute(), getNameForMembershipComparison(user)));

        String[] attributesToReturn = new String[]{searchProperties.getGroupnameAttribute()};
        LDAPPagerInfo result = search(searchFilter, attributesToReturn);

        if (UtilTimerStack.isActive())
            UtilTimerStack.pop(this.getClass().getName() + "_hasMembership(" + group + ", " + user + ")");

        return result.getNamingEnumeration().hasMoreElements();
    }

    private String getNameForMembershipComparison(User user)
    {
        return membershipProperties.isMembershipAttributeUnqualified() ? user.getName() :
            ((LDAPEntity) user).getDistinguishedName();
    }

    public LDAPPagerInfo getGroupEntriesViaMembership(User user) throws EntityException
    {
        if (!(user instanceof LDAPEntity))
        {
            log.info("Membership check for a non " + LDAPEntity.class.getName());
            return getGroupEntriesViaMembership(user.getName());
        }

        if (UtilTimerStack.isActive())
            UtilTimerStack.push(this.getClass().getName() + "_getLDAPGroupEntriesViaLDAPUser(" + user.getName() + ")");

        try
        {
            String[] attributesToReturn = new String[]{searchProperties.getGroupnameAttribute()};
            Filter searchTerm =
                new EqualsFilter(membershipProperties.getMembershipAttribute(), getNameForMembershipComparison(user));
            return getGroupEntries(attributesToReturn, searchTerm);

        }
        catch (RepositoryException e)
        {
            throw new RepositoryException(e);
        }
        finally
        {
            if (UtilTimerStack.isActive())
                UtilTimerStack.pop(
                    this.getClass().getName() + "_getLDAPGroupEntriesViaLDAPUser(" + user.getName() + ")");
        }

    }

    public LDAPPagerInfo getGroupEntriesViaMembership(String username) throws EntityException
    {
        return getGroupEntriesViaMembership(username, null);
    }

    public LDAPPagerInfo getGroupEntriesViaMembership(String username, String[] attributesToReturn)
        throws EntityException
    {
        if (UtilTimerStack.isActive())
            UtilTimerStack.push(this.getClass().getName() + "_getLDAPGroupEntriesViaMembership(" + username + ")");

        try
        {
            if (attributesToReturn == null)
                attributesToReturn = new String[]{searchProperties.getGroupnameAttribute()};

            String name = membershipProperties.isMembershipAttributeUnqualified() ?
                username : userAdaptor.getUserDN(username);
            Filter searchTerm = new EqualsFilter(membershipProperties.getMembershipAttribute(), name);
            return getGroupEntries(attributesToReturn, searchTerm);
        }
        catch (RepositoryException e)
        {
            throw new RepositoryException(e);
        }
        finally
        {
            if (UtilTimerStack.isActive())
                UtilTimerStack.pop(this.getClass().getName() + "_getLDAPGroupEntriesViaMembership(" + username + ")");
        }

    }
}