package com.atlassian.user.impl.hibernate.search.query;

import com.atlassian.user.EntityException;
import com.atlassian.user.impl.RepositoryException;
import com.atlassian.user.impl.hibernate.DefaultHibernateExternalEntity;
import com.atlassian.user.impl.hibernate.DefaultHibernateGroup;
import com.atlassian.user.impl.hibernate.DefaultHibernateUser;
import com.atlassian.user.impl.hibernate.repository.HibernateRepository;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.DefaultSearchResult;
import com.atlassian.user.search.SearchResult;
import com.atlassian.user.search.page.DefaultPager;
import com.atlassian.user.search.query.*;
import net.sf.hibernate.Criteria;
import net.sf.hibernate.HibernateException;
import net.sf.hibernate.Session;
import net.sf.hibernate.expression.*;
import org.springframework.orm.hibernate.SessionFactoryUtils;

import java.util.Iterator;
import java.util.List;

/**
 * Handles {@link Query} objects on a {@link HibernateRepository}.
 * <p/>
 * For now - boolean queries can only be performed on the same
 */
public class HibernateEntityQueryParser extends AbstractEntityQueryParser implements EntityQueryParser
{
    private final RepositoryIdentifier identifier;
    private final HibernateRepository repository;

    public HibernateEntityQueryParser(RepositoryIdentifier identifier, HibernateRepository repository)
    {
        this.identifier = identifier;
        this.repository = repository;
    }

    public SearchResult findUsers(Query query) throws EntityException
    {
        validateQuery(query);
        return parseQuery(query);
    }

    public SearchResult findGroups(Query query) throws EntityException
    {
        validateQuery(query);
        return parseQuery(query);
    }

    public SearchResult findUsers(Query query, QueryContext context) throws EntityException
    {
        validateQuery(query);

        if (context instanceof AllRepositoriesQueryContext ||
            context.getRepositoryKeys().contains(identifier.getKey()) ||
            context.getRepositoryKeys().contains(QueryContext.ALL_REPOSITORIES))
            return parseQuery(query);

        return null;
    }

    public SearchResult findGroups(Query query, QueryContext context) throws EntityException
    {
        validateQuery(query);

        if (context instanceof AllRepositoriesQueryContext ||
            context.getRepositoryKeys().contains(identifier.getKey()) ||
            context.getRepositoryKeys().contains(QueryContext.ALL_REPOSITORIES))

            return parseQuery(query);

        return null;
    }

    private MatchMode getMatchMode(String matchingRule)
    {
        if (matchingRule.equals(TermQuery.SUBSTRING_CONTAINS))
        {
            return MatchMode.ANYWHERE;
        }
        if (matchingRule.equals(TermQuery.SUBSTRING_ENDS_WITH))
        {
            return MatchMode.END;
        }
        if (matchingRule.equals(TermQuery.SUBSTRING_STARTS_WITH))
        {
            return MatchMode.START;
        }
        return MatchMode.EXACT;
    }

    private String identifyProperty(TermQuery q)
    {
        String entityProperty = null;

        if (q instanceof UserNameTermQuery)
            entityProperty = "name";
        else if (q instanceof EmailTermQuery)
            entityProperty = "email";
        else if (q instanceof FullNameTermQuery)
            entityProperty = "fullName";
        else if (q instanceof GroupNameTermQuery)
            entityProperty = "name";
        else if (q instanceof GroupsOfUserTwoTermQuery)
            entityProperty = "entity";

        return entityProperty;
    }

    /**
     * Converts a {@link BooleanQuery} object into an appropriate Hibernate {@link Criteria}, performs the search, and
     * returns a {@link SearchResult}.
     */
    private SearchResult parseQuery(Query q) throws EntityException
    {
        //1. establish the Object being queried.
        Session session = SessionFactoryUtils.getSession(repository.getSessionFactory(), true);
        Query definingQuery = q;

        if (q instanceof BooleanQuery)
            definingQuery = identifyDefiningQuery((BooleanQuery) q);

        List result;
        try
        {
            Criteria baseCriteria = getBaseCriteria(definingQuery, session);
            baseCriteria = identifyAndAddSearchCriteria(q, definingQuery, baseCriteria);

            result = baseCriteria.list();
        }
        catch (HibernateException e)
        {
            throw new RepositoryException(e);
        }

        return new DefaultSearchResult(new DefaultPager(result), identifier.getName());
    }

    private Criteria identifyAndAddSearchCriteria(Query q, Query definingQuery, Criteria baseCriteria)
        throws EntityQueryException, HibernateException
    {
        if (q instanceof BooleanQuery)
            return addSearchCriteria((BooleanQuery) q, definingQuery, baseCriteria);
        else
        {
            return addSearchCriteria((TermQuery) q, baseCriteria);
        }
    }

    private Criteria addSearchCriteria(BooleanQuery q, Query definingQuery, Criteria baseCriteria)
        throws EntityQueryException, HibernateException
    {
        /**
         * Membership queries are performed on associations - i.e. subCriteria, and need to be handled on a case by case basis.
         */
        if (definingQuery instanceof MembershipQuery)
        {
            return addMembershipSearchCriteria(q, baseCriteria);
        }

        Junction junction = identifyAndOrJunction(q);
        baseCriteria.add(junction);
        Iterator iter = q.getQueries().iterator();

        while (iter.hasNext())
        {
            Query query = (Query) iter.next();

            if (query instanceof BooleanQuery)
            {
                addSearchCriteria((BooleanQuery) query, definingQuery, baseCriteria);
            }
            else if (query instanceof TermQuery)
            {
                junction.add(getQueryExpression((TermQuery) query));
            }
            else
                throw new EntityQueryException("Unknown query type: [" + query.getClass().getName() + "]");
        }

        return baseCriteria;
    }

    private Junction identifyAndOrJunction(BooleanQuery q)
    {
        Junction junction;
        if (q.isAND())
            junction = Expression.conjunction();
        else
            junction = Expression.disjunction();

        return junction;
    }

    private Criteria addMembershipSearchCriteria(BooleanQuery q, Criteria baseCriteria) throws HibernateException
    {
        if (q instanceof GroupsOfUserTwoTermQuery)
        {
            addGroupsOfUserSearchCriteria(q, baseCriteria);
        }
        else if (q instanceof GroupsOfExternalEntityTwoTermQuery)
        {
            addGroupsOfExternalEntitySearchCriteria(q, baseCriteria);
        }

        return baseCriteria;
    }

    private void addGroupsOfUserSearchCriteria(BooleanQuery q, Criteria baseCriteria) throws HibernateException
    {
        UserNameTermQuery userNameQuery = ((GroupsOfUserTwoTermQuery) q).getUserNameTermQuery();
        GroupNameTermQuery groupNameQuery = ((GroupsOfUserTwoTermQuery) q).getGroupNameTermQuery();

        if (groupNameQuery.getTerm().equals(TermQuery.WILDCARD))
        {
            //do nothing, there is no specific group query and we defer to the username term below
        }
        else if (groupNameQuery.isMatchingSubstring())
        {
            baseCriteria.add(getLikeExpression("name", groupNameQuery, false));
        }
        else
        {
            baseCriteria.add(new EqExpression("name", groupNameQuery.getTerm(), false));
        }

        if (userNameQuery.isMatchingSubstring())
        {
            baseCriteria.createCriteria("localMembers").add(getLikeExpression("name", userNameQuery, false));
        }
        else
        {
            baseCriteria.createCriteria("localMembers").add(new EqExpression("name", userNameQuery.getTerm(), false));
        }
    }

    private Criterion getLikeExpression(String entityAttribute, TermQuery termQuery, boolean caseInsensitive)
    {
        if (caseInsensitive)
            return Expression.ilike(entityAttribute, termQuery.getTerm(), getMatchMode(termQuery.getMatchingRule()));
        else
            return Expression.like(entityAttribute, termQuery.getTerm(), getMatchMode(termQuery.getMatchingRule()));
    }

    private void addGroupsOfExternalEntitySearchCriteria(BooleanQuery q, Criteria baseCriteria)
        throws HibernateException
    {
        ExternalEntityNameTermQuery nameQuery =
            ((GroupsOfExternalEntityTwoTermQuery) q).getExternalEntityNameTermQuery();
        GroupNameTermQuery groupQuery = ((GroupsOfExternalEntityTwoTermQuery) q).getGroupNameTermQuery();

        if (groupQuery.getTerm().equals(TermQuery.WILDCARD))
        {
            //do nothing, there is no specific group query and we defer to the username term below
        }
        else if (groupQuery.isMatchingSubstring())
        {
            baseCriteria.add(getLikeExpression("name", groupQuery, false));
        }
        else
        {
            baseCriteria.add(new EqExpression("name", groupQuery.getTerm(), false));
        }

        if (nameQuery.isMatchingSubstring())
        {
            baseCriteria.createCriteria("externalMembers").add(getLikeExpression("name", nameQuery, false));
        }
        else
        {
            baseCriteria.createCriteria("externalMembers").add(new EqExpression("name", nameQuery.getTerm(), false));
        }
    }

    private Criteria addSearchCriteria(TermQuery q, Criteria baseCriteria)
    {
        Criterion expression = getQueryExpression(q);
        baseCriteria.add(expression);
        return baseCriteria;
    }

    private Criterion getQueryExpression(TermQuery termQuery)
    {
        String hqlField = identifyProperty(termQuery);

        if (termQuery.isMatchingSubstring())
        {
            return getLikeExpression(hqlField, termQuery, true);
        }
        else
        {
            return new EqExpression(hqlField, termQuery.getTerm(), true);
        }
    }

    private Criteria getBaseCriteria(Query query, Session session)
    {
        Criteria baseCriteria = null;

        if (query instanceof UserQuery)
        {
            baseCriteria = session.createCriteria(DefaultHibernateUser.class);
        }
        else if (query instanceof GroupQuery)
        {
            baseCriteria = session.createCriteria(DefaultHibernateGroup.class);
        }
        else if (query instanceof UsersInGroupTwoTermQuery)
        {
            baseCriteria = session.createCriteria(DefaultHibernateUser.class);
        }
        else if (query instanceof GroupsOfUserTwoTermQuery || query instanceof GroupsOfExternalEntityTwoTermQuery)
        {
            baseCriteria = session.createCriteria(DefaultHibernateGroup.class);
        }
        else if (query instanceof ExternalEntitiesInGroupTwoTermQuery)
        {
            baseCriteria = session.createCriteria(DefaultHibernateExternalEntity.class);
        }

        return baseCriteria;
    }
}
