package com.atlassian.user.impl.cache;

import com.atlassian.user.Group;

/**
 * Caches the groups retrieved from the underlying group manager.
 * <br/>
 * Will also cache the fact that a group could not be found so that we don't incur
 * the expense of another search when have previously determined that a group doesn't
 * exist.
 *
 * @see CachingGroupManager
 */
public class GroupCache
{
    private final CacheManager cacheManager;
    private final String cacheName;

    /**
     * The NULL_GROUP is stored as the value in the groups cache when
     * a group that does not exist is requested.
     */
    public static final Group NULL_GROUP = new Group()
    {
        private String NAME = "NULL GROUP";

        public String getName()
        {
            return NAME;
        }

        public boolean equals(Object obj)
        {
            return obj.getClass().getName().equals(getClass().getName());
        }

        public int hashCode()
        {
            return getClass().getName().hashCode();
        }
    };

    public GroupCache(CacheManager cacheManager, String cacheName)
    {
        this.cacheManager = cacheManager;
        this.cacheName = cacheName;
    }

    private Cache getCache()
    {
        return cacheManager.getCache(cacheName);
    }

    /**
     * Stores the specified group in the cache against the group name.
     * If group is null, stores {@link #NULL_GROUP} in the cache so the
     * fact that the group does not exist is remembered by the
     * cache.
     */
    public void put(String groupName, Group group)
    {
        getCache().put(groupName, group == null ? NULL_GROUP : group);
    }

    /**
     * @return the group with the cached name, {@link #NULL_GROUP} if the
     * group could not be found on the previous search, or null if no
     * group with that name is in the cache.
     */
    public Group get(String groupName)
    {
        return (Group) getCache().get(groupName);
    }

    public void remove(String groupName)
    {
        getCache().remove(groupName);
    }
}
