UPM.define('ApplicationCollectionItemView', [
    'jquery',
    'underscore',
    'AuiFlagManager',
    'BaseCollectionItemView',
    'BlanketableViewMixin',
    'InlineDialogViewMixin',
    'ManageApplicationsEnvironment',
    'MultipleLicenseUpdateDialog',
    'NotificationCollectionView',
    'PlatformUpdateDialog',
    'CustomerAgreementDialog',
    'UpmAjax',
    'UpmDialog',
    'UninstallDialog',
    'UnlicenseDialog',
    'UpmEnvironment',
    'UpmAnalytics'
], function($,
            _,
            AuiFlagManager,
            BaseCollectionItemView,
            BlanketableViewMixin,
            InlineDialogViewMixin,
            ManageApplicationsEnvironment,
            MultipleLicenseUpdateDialog,
            NotificationCollectionView,
            PlatformUpdateDialog,
            CustomerAgreementDialog,
            UpmAjax,
            UpmDialog,
            UninstallDialog,
            UnlicenseDialog,
            UpmEnvironment,
            UpmAnalytics) {


    'use strict';


    return BaseCollectionItemView.extend({

        mixins: [ BlanketableViewMixin, InlineDialogViewMixin ],

        events: {
            'click .update-license-key': '_showEditableLicenseKey',
            'click .license-string-truncated': '_showEditableLicenseKey',
            'click .paste-platform-license': '_showEditableLicenseKey',
            'click .delete-license-key': '_deleteLicenseKey',
            'click .license-update-cancel': '_cancelLicenseUpdate',
            'click .submit': '_submitLicenseUpdate',
            'submit .license-update-form': '_submitLicenseUpdate',
            'click .uninstall-application-plugin': '_uninstallApplicationPlugin',
            'keyup textarea': '_onLicenseTextChange',
            'input textarea': '_onLicenseTextChange',
            'propertychange textarea': '_onLicenseTextChange',
            'click .trial-notification-action': '_evaluateApplication',
            'click .buy-notification-action': '_purchaseApplication',
            'click .download-platform-notification-action': '_downloadPlatformApplication',
            'click .renew-notification-action': '_renewApplication',
            'click .update-notification-action': '_updateApplication',
            'click .upgrade-notification-action': '_upgradeApplication',
            'click .install-notification-action': '_installApplication'
        },

        _initEvents: function() {
            this.listenTo(this.model, 'change:progressValue', this._updateLoadingPercentage);
            this.listenTo(this.model, 'change:loading', this._changeInlineProgress);
            BaseCollectionItemView.prototype._initEvents.apply(this);
        },

        _postInitialize: function () {
            this.render();
        },

        /**
         * Render function uses the soy template to generate the updated view
         */
        template: com.atlassian.upm.application.applicationRow,

        _getData: function () {
            return {
                app: this.model.toJSON(),
                mpacUnreachable: UpmEnvironment.isMpacUnavailable(),
                mpacDisabled: UpmEnvironment.isMpacDisabled()
            };
        },

        _renderTitleBar: function () {
            var newTitleBar = com.atlassian.upm.application.applicationRowTitle({
                app: this.model.toJSON()
            });
            this.$el.find('.application-title').replaceWith(newTitleBar);
        },

        _postRender: function () {
            new NotificationCollectionView({
                collection: this.model.getNotifications(),
                el: this.$el.find('.application-notifications')
            }).render();
            this._initMaintenanceExpiresInlineDialog();
            this._initUserCountDetailsInlineDialog();
            this._initPlatformDetailsInlineDialog();
        },

        _updateLoadingPercentage: function () {
            var meter = this.$el.find('.app-loading-progress-bar');
            if (meter.length > 0) {
                AJS.progressBars.update(meter, this.model.getProgressValue());
            }
        },

        /**
         * Callback when the model `loading` changes. When loading turns on, undelegate
         * all events so the user can't try to do things while this app is busy. When loading
         * turns back to false, re-render the entire view and re-enable events.
         */
        _changeInlineProgress: function () {
            if (this.model.getLoading()) {
                if (!this.model.getProgressMessage()) {
                    this.model.setProgressMessage(AJS.I18n.getText('upm.applications.loading'));
                }
                this._renderTitleBar();
                this.setBlanketed(true, '.application-content');
                this.undelegateEvents();

            } else {
                this.model.setProgressValue(1.0);
                setTimeout(_.bind(function () {
                    this.setBlanketed(false, '.application-content');
                    this.render();
                    this.delegateEvents();
                }, this), 350);
            }
        },

        _downloadPlatformApplication: function (e) {
            var that = this;
            e && e.preventDefault && e.preventDefault();
            if (this.model.getPlatform() && this.model.getDownloadsPageUri()) {
                this.model.startLoading();
                this.model.findAppUpdatesForPlatform()
                    .fail(function(request) {
                        UpmAjax.signalAjaxError(request);
                    })
                    .done(function(resp) {
                        var updatableApps = resp.updates;
                        new PlatformUpdateDialog({
                            model: that.model,
                            updatableApps: updatableApps
                        }).getResult()
                            .done(function(appKeysToUpdate) {
                                // We have to open this new window immediately, rather than in a .done() handler for
                                // the Ajax call, because otherwise popup blockers will intercept it.
                                window.open(that.model.getDownloadsPageUri());
                                return that.model.scheduleAppUpdatesForPlatform(
                                    _.filter(updatableApps, function(a) {
                                        return _.contains(appKeysToUpdate, a.key);
                                    }))
                                    .fail(UpmAjax.signalAjaxError);
                            });
                    })
                    .always(function() {
                        that.model.stopLoading();
                    });
            }
        },

        /**
         * Prompts user for AtlassianID credentials (if not already stored) then makes a request to
         * Hamlet/MAC/FakePAC to get an evaluation license for the product
         * But if it's a Data Center instance, show a flag instead: https://ecosystem.atlassian.net/browse/UPM-5027
         * @private
         */
        _evaluateApplication: function (e) {
            e && e.preventDefault && e.preventDefault();

            UpmAnalytics.logEvent('manageapps-try', { appKey: this.model.getKey() });

            if(ManageApplicationsEnvironment.isDataCenter()) {
                AuiFlagManager.flag({
                   type: 'info',
                   body: AJS.I18n.getText('upm.applications.notifications.flag.evaluation.datacenter', this.model.getName(), AJS.I18n.getText('upm.applications.notifications.datacenter.suffix')),
                   close: 'manual'
                });
            } else {
                var aidOptions = {
                    applicationFormat: true,
                    applicationName: this.model.getName()
                };
                var me = this;

                return new CustomerAgreementDialog({ showRedirectMessage: true }).getResult().then(
                    function() {
                        return me.model.startEvaluation();
                    }
                ).then(
                    _.bind(function () {
                        var postInstallLink = this.model.getLinks()['post-install'];
                        AuiFlagManager.delayedFlag({
                            type: 'success',
                            body: (postInstallLink ?
                                   AJS.I18n.getText('upm.applications.flags.license.added.postinstall', this.model.getName(), postInstallLink) :
                                   AJS.I18n.getText('upm.applications.flags.license.added', this.model.getName())),
                            close: (postInstallLink ? 'manual' : 'auto')
                        });
                        UPM.trace('eval-success');
                    }, this)
                ).fail(
                    function (xhr, status, msg) {
                        if (msg) {
                            UpmAjax.getEventTarget().trigger('ajaxError', msg);
                        }
                    }
                );
            }
        },

        /**
         * Sends the user towards a purchase flow
         */
        _purchaseApplication: function (e) {
            e && e.preventDefault && e.preventDefault();
            var data = { appKey: this.model.getKey() };
            if (this.model.getAccessDetails()) {
                data.unitCount = this.model.getAccessDetails().getActiveUserCount();
            }
            UpmAnalytics.logEvent('manageapps-buy', data).always(_.bind(function () {
                window.location.href = this.model.getLinks().buy;
            }, this));
        },

        _renewApplication: function (e) {
            e && e.preventDefault && e.preventDefault();
            UpmAnalytics.logEvent('manageapps-renew', {
                appKey: this.model.getKey(),
                daysToExpiry: this.model.getLicense().daysUntilExpiry()
            }).always(_.bind(function () {
                window.location.href = this.model.getLinks().renew;
            }, this));
        },

        _updateApplication: function(e) {
            e.preventDefault();
            UpmAnalytics.logEvent('manageapps-update', {
                appKey: this.model.getKey(),
                updateVrs: this.model.getUpdate().updateVersion
            });
            var releaseNotes = this.model.getUpdateReleaseNotes();
            this.model.updateApplication().then(
                _.bind(function () {
                    AuiFlagManager.flag({
                        type: 'success',
                        body: (releaseNotes ?
                            AJS.I18n.getText('upm.applications.flags.update.success.releasenotes', this.model.getName(), releaseNotes) :
                            AJS.I18n.getText('upm.applications.flags.update.success', this.model.getName())
                        ),
                        close: (releaseNotes ? 'manual' : 'auto')
                    });
                }, this)
            ).fail(
                function (xhr, status, message) {
                    UpmAjax.getEventTarget().trigger('ajaxError', message);
                }
            );
        },

        _upgradeApplication: function (e) {
            e && e.preventDefault && e.preventDefault();
            UpmAnalytics.logEvent('manageapps-upgrade', {
                appKey: this.model.getKey(),
                remainingUsers: this.model.getAccessDetails().remainingUsers()
            }).always(_.bind(function () {
                window.location.href = this.model.getLinks().upgrade;
            }, this));
        },

        /**
         * Modify license key
         */
        _showEditableLicenseKey: function(e) {
            e && e.preventDefault();
            if (this.model.getLicensedButNotInstalled()) {
                return;
            }
            this._toggleEditLicenseDisplay(true);
        },

        _cancelLicenseUpdate: function(e) {
            e.preventDefault();
            this.$el.find('.license-update-form textarea').val(this.model.getLicenseKey());
            this._toggleEditLicenseDisplay(false);
        },

        _toggleEditLicenseDisplay: function(showEditable) {
            if (this.model.isLicensed()) {
                this.$el.find('#license-string-collapsed-view').toggleClass('hidden', showEditable);
            }
            this.$el.find('.license-update-form').toggleClass('hidden', !showEditable);
            this.$el.find('.paste-platform-license').toggleClass('hidden', showEditable);
            if (showEditable) {
                this.$el.find('.license-update-form textarea').focus().select();
            }
        },

        _onLicenseTextChange: function(e) {
            var $textarea = this.$el.find('.license-update-form textarea'),
                $submit = this.$el.find('.license-update-form .submit'),
                textValue = $textarea.val().trim(),
                unChanged = textValue.replace(/\r\n/g, '\n') === this.model.getLicenseKey().replace(/\r\n/g, '\n'),
                disabled = unChanged || textValue.length === 0;

            if (e && e.keyCode) {
                if (e.keyCode === 13) { // enter will submit
                    e.preventDefault();
                    if (!disabled) {
                        this._submitLicenseUpdate();
                    }
                } else if (e.keyCode === 27) { // esc will cancel
                    this._toggleEditLicenseDisplay(false);
                }
            }

            $submit.prop('disabled', disabled);
        },

        /**
         * Submits the user-entered licenseKey to the server.
         */
        _submitLicenseUpdate: function(e) {
            e && e.preventDefault && e.preventDefault();
            UpmAnalytics.logEvent('manageapps-submit-license', { appKey: this.model.getKey() });
            var self = this,
                licenseKey = this.$el.find('.license-update-form textarea').val(),
                licensedPreviously = !!self.model.getLicense().getRawLicense();
            this.$el.find('.license-update-form textarea, .license-update-form input').attr('disabled', 'disabled');
            this.model.submitLicense(licenseKey).then(function (resp) {
                if (resp.validation.warningMessages.length) {
                    AuiFlagManager.delayedFlag({
                       type: 'warning',
                       body: _.first(resp.validation.warningMessages),
                       close: 'manual'
                    });
                    UpmAnalytics.logEvent('manageapps-submit-license-warning', {
                        appKey: self.model.getKey(),
                        licenseType: self.model.getLicense().getLicenseType()
                    });
                } else {
                    if (licensedPreviously) {
                        AuiFlagManager.delayedFlag({
                           type: 'success',
                           body: AJS.I18n.getText('upm.applications.flags.license.updated', self.model.getName()),
                           close: 'auto'
                        });
                    } else {
                        var postInstallLink = self.model.getLinks()['post-install'];
                        AuiFlagManager.flag({
                            type: 'success',
                            body: (postInstallLink ?
                                AJS.I18n.getText('upm.applications.flags.license.added.postinstall', self.model.getName(), postInstallLink) :
                                AJS.I18n.getText('upm.applications.flags.license.added', self.model.getName())),
                            close: postInstallLink ? 'manual' : 'auto',
                            delay: 300
                        });
                    }
                    UpmAnalytics.logEvent('manageapps-submit-license-success', {
                        appKey: self.model.getKey(),
                        licenseType: self.model.getLicense().getLicenseType()
                    });
                }
            }).fail(
                function (xhr, status, msg) {
                    if (xhr.responseJSON.licenseMismatch) {
                        var dialog = new MultipleLicenseUpdateDialog({
                            data: xhr.responseJSON.licenseMismatch,
                            model: self.model,
                            page: 1
                        });
                        dialog.getResult();
                    } else {
                        msg = AJS.I18n.getText('upm.applications.license.installation.fail') + '<br><br>' + msg;
                        UpmAjax.getEventTarget().trigger('ajaxError', msg);
                    }
                    UpmAnalytics.logEvent('manageapps-submit-license-failure');
                }
            );
        },


        /**
         * Deletes the licenseKey from the server.
         */
        _deleteLicenseKey: function(e) {
            e && e.preventDefault && e.preventDefault();
            UpmAnalytics.logEvent('manageapps-delete-license', { appKey: this.model.getKey() });
            var self = this;

            var dialog = new UnlicenseDialog({
                data: {
                    app: this.model.toJSON()
                }
            });

            dialog.getResult().done(function(response) {
                self.$el.find('.license-update-form textarea, .license-update-form input').attr('disabled', 'disabled');
                self.model.deleteLicenseAndConfiguration(response.deleteConfig).then(function () {
                    AuiFlagManager.delayedFlag({
                        type: 'success',
                        body: AJS.I18n.getText('upm.applications.flags.license.removed', self.model.getName()),
                        close: 'auto'
                    });
                }).always(
                    _.bind(function () {
                        self.model.stopLoading();
                    }, this)
                ).fail(
                    function (xhr, status, msg) {
                        UpmAjax.getEventTarget().trigger('ajaxError', msg);
                    }
                );
            });
        },

        _initMaintenanceExpiresInlineDialog: function() {
            var content = com.atlassian.upm.application.maintenanceExpiresInlineDialog({
                app: UpmEnvironment.getApplicationName()
            });
            if (this.model.isLicensed()) {
                this.inlineDialog('.maintenance-expires-info-icon', 'maintenanceExpiresDialog', content);
            }
        },

        _initUserCountDetailsInlineDialog: function() {
            var content = com.atlassian.upm.application.userCountDetailsInlineDialog({
                platformName: UpmEnvironment.getApplicationName()
            });
            if (this.model.isLicensed()) {
                this.inlineDialog('.user-count-details-icon', 'userCountDetailsDialog', content);
            }
        },

        _initPlatformDetailsInlineDialog: function () {
            var content = com.atlassian.upm.application.platformDetailsInlineDialog({
                platformName: UpmEnvironment.getApplicationName()
            });
            if (this.model.isPlatform() && !this.model.isLicensed()) {
                this.inlineDialog('.platform-application-icon', 'platformDetailsHover', content);
            }
        },

        _installApplication: function (e) {
            e.preventDefault();
            if (!UpmEnvironment.isMpacAvailable() || !(this.model.getLinks() && this.model.getLinks().binary)) {
                window.location.href = this.model.getLinks()['wac-update'];
                return;
            }

            this.model.installFromBinaryUri().then(
                _.bind(function () {
                    var postInstallLink = this.model.getLinks()['post-install'];
                    AuiFlagManager.flag({
                        type: 'success',
                        body: (postInstallLink ?
                            AJS.I18n.getText('upm.applications.flags.installation.success.postinstall', this.model.getName(), postInstallLink) :
                            AJS.I18n.getText('upm.applications.flags.installation.success', this.model.getName())),
                        close: postInstallLink ? 'manual' : 'auto',
                        delay: 300
                    });
                }, this)
            ).fail(
                function (xhr, status, message) {
                    UpmAjax.getEventTarget().trigger('ajaxError', message);
                }
            ).always(
                _.bind(function () {
                    this.model.stopLoading();
                }, this)
            );
        },

        _uninstallApplicationPlugin: function(e) {
            var that = this,
                dialog;
            e.preventDefault();
            dialog = new UninstallDialog({
                data: {
                    app: this.model.toJSON(),
                    license: (this.model.isLicensed() ? this.model.getLicense().toJSON() : null)
                }
            });
            dialog.getResult().done(function(response) {
                var updateProgressCallback = _.bind(that.model.setProgressValue, that.model);
                that.model.startLoading(AJS.I18n.getText('upm.applications.uninstalling'));
                that.model.uninstall(updateProgressCallback, response.deleteLicenseAndConfig)
                    .done(function () {
                        AuiFlagManager.flag({
                            type: 'success',
                            body: AJS.I18n.getText('upm.applications.flags.uninstall.success', that.model.getName()),
                            close: 'auto'
                        });
                    })
                    .fail(function (xhr, status, msg) {
                        if (msg) {
                            UpmAjax.getEventTarget().trigger('ajaxError', msg);
                        }
                    });
            });
        }
    });
});
