package com.atlassian.upm.api.log;

/**
 * Represents various types of audit log entries.
 *
 * @since 1.6
 */
public enum EntryType
{
    PLUGIN_INSTALL("upm.auditLog.entryType.pluginInstall", "upm.auditLog.install"),
    PLUGIN_UNINSTALL("upm.auditLog.entryType.pluginUninstall", "upm.auditLog.uninstall"),
    PLUGIN_ENABLE("upm.auditLog.entryType.pluginEnable", "upm.auditLog.enable"),
    PLUGIN_DISABLE("upm.auditLog.entryType.pluginDisable", "upm.auditLog.disable"),
    // The key containing "upgrade" will no longer be produced by the UPM -- we'll use "update" instead. But this is included for backward compatibility.
    PLUGIN_UPDATE("upm.auditLog.entryType.pluginUpdate", "upm.auditLog.update", "upm.auditLog.upgrade"),
    AUTO_PLUGIN_UPDATE("upm.auditLog.entryType.pluginAutoUpdate", "upm.auditLog.auto.update"),
    ENTER_SAFE_MODE("upm.auditLog.entryType.enterSafeMode", "upm.auditLog.safeMode.enter"),
    EXIT_SAFE_MODE("upm.auditLog.entryType.exitSafeMode", "upm.auditLog.safeMode.exit"),
    // This entry type is deprecated in favor of UPM_STARTUP; it will no longer be produced by the UPM
    SYSTEM_STARTUP("upm.auditLog.entryType.systemStartup", "upm.auditLog.system.startup"),
    UPM_STARTUP("upm.auditLog.entryType.upmStartup", "upm.auditLog.upm.startup"),
    CANCELLED_CHANGE("upm.auditLog.entryType.cancelledChange", "upm.auditLog.cancelChange"),
    LICENSE_ADD("upm.auditLog.entryType.licenseAdd", "upm.auditLog.plugin.license.add"),
    LICENSE_REMOVE("upm.auditLog.entryType.licenseRemove", "upm.auditLog.plugin.license.remove"),
    LICENSE_UPDATE("upm.auditLog.entryType.licenseUpdate", "upm.auditLog.plugin.license.update"),
    UNCLASSIFIED_EVENT("upm.auditLog.entryType.unclassified", null);

    private final String i18nName;
    private final String[] i18nPrefixes;

    private EntryType(String i18nName, String... i18nPrefixes)
    {
        this.i18nName = i18nName;
        this.i18nPrefixes = i18nPrefixes;
    }

    /**
     * Returns a i18n key for a displayable name.
     * @return a i18n key for a displayable name.
     */
    public String getI18nName()
    {
        return i18nName;
    }

    /**
     * Returns the i18n key prefixes for which all audit log entries of this type will start with.
     * @return the i18n key prefixes for which all audit log entries of this type will start with.
     */
    private String[] getI18nPrefixes()
    {
        return i18nPrefixes;
    }
    
    private static boolean startsWithI18nPrefix(String i18n, EntryType type)
    {
        if (type.getI18nPrefixes() != null)
        {
            for (String prefix : type.getI18nPrefixes())
            {
                if (i18n.startsWith(prefix))
                {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Returns the {@link EntryType} associated with the given i18n key
     * @param i18n the i18n key
     * @return the {@link EntryType} associated with the given i18n key
     */
    public static EntryType valueOfI18n(String i18n)
    {
        for (EntryType type : values())
        {
            if (startsWithI18nPrefix(i18n, type))
            {
                return type;
            }
        }
        return UNCLASSIFIED_EVENT;
    }
}
