package com.atlassian.upm.license.role.spi;

import com.atlassian.plugin.Plugin;

/**
 * {@link com.atlassian.upm.license.role.spi.RoleBasedLicenseService} is the link between role-based licensing
 * SPI implementations and the UPM. If present, UPM will use {@link com.atlassian.upm.license.role.spi.RoleBasedLicenseService}
 * to properly manage and report role-based licensed plugins.
 *
 * Each application should provide their own {@link com.atlassian.upm.license.role.spi.RoleBasedLicenseService} implementation
 * based on their backing mechanism. For example, JIRA's implementation would be based on pluggable global permissions.
 *
 * Please note that UPM does not perform any role caching. Performance enhancements must be added to the SPI implementations as desired.
 *
 * UPM does, however, cache plugin licenses when appropriate. {@link com.atlassian.upm.license.role.spi.RoleBasedLicenseService}
 * implementations should be sure to fire {@link com.atlassian.upm.license.role.spi.LicensingRoleMembershipUpdatedEvent}s
 * when users/groups are added or removed from a licensing role to notify UPM to invalidate its cache.
 */
public interface RoleBasedLicenseService
{
    /**
     * If a licensing role with the given role key already exists, it is returned without creating or modifying any roles.
     * If no licensing role can be found with the given role key, a new one is created.
     *
     * Licensing roles may or may not use the {@code nameI18nKey} and {@code descriptionI18nKey} parameters depending
     * on the application's licensing role implementation.
     *
     * Implementations can assume that this method will only be invoked for plugins which are enabled and licensed.
     *
     * @param plugin the plugin
     * @param key the role key
     * @param nameI18nKey an i18n key for the role name
     * @param descriptionI18nKey an i18n key for the description name
     * @return the licensing role
     * @throws LicensingRoleCreationFailedException if an attempted licensing role creation fails for any reason.
     */
    LicensingRole createLicensingRole(Plugin plugin, String key, String nameI18nKey, String descriptionI18nKey) throws LicensingRoleCreationFailedException;

    /**
     * Returns the licensing role with the given role key, or null if no such role exists.
     *
     * @param plugin the plugin
     * @param key the role key
     * @return the licensing role with the given role key, or null if no such role exists.
     */
    LicensingRole getLicensingRole(Plugin plugin, String key);

    /**
     * Handle the event where the given plugin becomes unlicensed. This allows the service to clean up
     * after role-based plugins when they're unlicensed.
     *
     * @param plugin the plugin
     */
    void onPluginUnlicensedEvent(Plugin plugin);
}
