package com.atlassian.upm.license.role.spi;

import java.net.URI;

/**
 * Represents a licensing role for a role-based licensed plugin. This role defines the mechanism with which
 * the plugin will be licensed.
 *
 * {@link com.atlassian.upm.license.role.spi.LicensingRole} data is populated from the backing implementation.
 * For example, in JIRA, {@link com.atlassian.upm.license.role.spi.LicensingRole}s may be backed by
 * pluggable global permissions.
 *
 * In order to have a licensing role, plugins needs to declare "atlassian-licensing-role-based-enabled" and
 * "atlassian-licensing-enabled" params in their plugin descriptor's "<plugin-info>" tag.
 */
public interface LicensingRole
{
    /**
     * Returns the key for this licensing role. This is the role's primary identifier used for lookup purposes. It must
     * adhere to the key naming restrictions imposed by the host application and the SPI implementation.
     *
     * This should <strong>never</strong> return null.
     *
     * The licensed plugin may specify this key with the "atlassian-licensing-role-based-enabled" param in a
     * plugin descriptor's "<plugin-info>" tag. UPM will provide a default value if unspecified.
     */
    String getKey();

    /**
     * Returns the i18n key for this licensing role's name. This name may or may not actually be used depending on the
     * SPI implementation.
     *
     * If the implementation which backs the license role does not contain or use a name, this will return null.
     *
     * The licensed plugin may specify this key with the "atlassian-licensing-role-name-key" param in a
     * plugin descriptor's "<plugin-info>" tag. UPM will provide a default value if unspecified, although whether or
     * not the value is used is up to the SPI implementation.
     *
     * @return the i18n key for the name property, or null if this role does not have a name.
     */
    String getNameI18nKey();

    /**
     * Returns the i18n key for this licensing role's description. This description may or may not actually be used depending on the
     * SPI implementation.
     *
     * If the implementation which backs the license role does not contain or use a description, this will return null.
     *
     * The licensed plugin may specify this key with the "atlassian-licensing-role-description-key" param in a
     * plugin descriptor's "<plugin-info>" tag. UPM will provide a default value if unspecified, although whether or
     * not the value is used is up to the SPI implementation.
     *
     * @return the i18n key for the name property, or null if this role does not have a name.
     */
    String getDescriptionI18nKey();

    /**
     * Returns a relative URI to a screen where admins can manage users in this role. The URI must reference a page
     * on the current application instance. This URI should not include the application's context path if one exists.
     */
    URI getManagementPage();

    /**
     * Returns true if the given user has membership in this licensing role, false otherwise. Returns false
     * if no user exists for the given user key.
     *
     * This function will be called frequently; it is strongly recommended that implementations use a caching strategy
     * to reduce the cost of repeated calls.
     *
     * @param userKey the user key (<strong>not</strong> the username)
     * @return true if the given user has membership in this licensing role, false otherwise
     */
    boolean isUserInRole(String userKey);

    /**
     * Returns the number of users in this licensing role.
     *
     * This function will be called frequently; it is strongly recommended that implementations use a caching strategy
     * to reduce the cost of repeated calls.
     *
     * @return the number of users in this licensing role.
     */
    int getRoleCount();
}
