package com.atlassian.upm.license.role.jira;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import com.atlassian.crowd.event.DirectoryEvent;
import com.atlassian.crowd.event.directory.RemoteDirectorySynchronisedEvent;
import com.atlassian.crowd.event.group.GroupMembershipCreatedEvent;
import com.atlassian.crowd.event.group.GroupMembershipDeletedEvent;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.jira.event.ClearCacheEvent;
import com.atlassian.jira.event.permission.AbstractGlobalPermissionEvent;
import com.atlassian.jira.permission.GlobalPermissionKey;
import com.atlassian.jira.permission.GlobalPermissionType;
import com.atlassian.jira.security.GlobalPermissionManager;
import com.atlassian.jira.user.util.UserManager;
import com.atlassian.plugin.Plugin;
import com.atlassian.pocketknife.api.lifecycle.modules.ModuleRegistrationHandle;
import com.atlassian.upm.license.role.spi.LicensingRole;
import com.atlassian.upm.license.role.spi.LicensingRoleCreationFailedException;
import com.atlassian.upm.license.role.spi.LicensingRoleMembershipUpdatedEvent;
import com.atlassian.upm.license.role.spi.RoleBasedLicenseService;

import com.google.common.collect.ImmutableList;

import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;

import static com.atlassian.fugue.Option.option;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * JIRA-specific implementation of the role-based licensing SPI. This implementation is based on JIRA's
 * use of pluggable global permissions.
 */
public class JiraRoleBasedLicenseService implements RoleBasedLicenseService, InitializingBean, DisposableBean
{
    private final GlobalPermissionManager permissionManager;
    private final UserManager userManager;
    private final EventPublisher eventPublisher;
    private final GlobalPermissionUserCountCache globalPermissionUserCountCache;
    private final DynamicGlobalPermissionService dynamicGlobalPermissionService;

    private final ConcurrentMap<String, ConcurrentMap<String, String>> pluginKeysPerPermission;
    private final ConcurrentMap<String, ModuleRegistrationHandle> permissionModulesPerPluginKey;

    public JiraRoleBasedLicenseService(GlobalPermissionManager permissionManager,
                                       UserManager userManager,
                                       EventPublisher eventPublisher,
                                       DynamicGlobalPermissionService dynamicGlobalPermissionService,
                                       GlobalPermissionUserCountCache globalPermissionUserCountCache)
    {
        this.permissionManager = checkNotNull(permissionManager, "permissionManager");
        this.userManager = checkNotNull(userManager, "userManager");
        this.eventPublisher = checkNotNull(eventPublisher, "eventPublisher");
        this.dynamicGlobalPermissionService = checkNotNull(dynamicGlobalPermissionService, "dynamicGlobalPermissionService");
        this.globalPermissionUserCountCache = checkNotNull(globalPermissionUserCountCache, "globalPermissionUserCountCache");

        this.pluginKeysPerPermission = new ConcurrentHashMap<String, ConcurrentMap<String, String>>();
        this.permissionModulesPerPluginKey = new ConcurrentHashMap<String, ModuleRegistrationHandle>();
    }

    @Override
    public LicensingRole createLicensingRole(Plugin plugin, String key, String nameI18nKey, String descriptionI18nKey) throws LicensingRoleCreationFailedException
    {
        // Try and see if the role exists first
        for (LicensingRole role : option(getLicensingRole(plugin, key)))
        {
            return role;
        }

        // Create the global permission and append its plugin module to the relevant PvA plugin
        ModuleRegistrationHandle moduleHandle = dynamicGlobalPermissionService.dynamicallyCreatePermission(plugin, key, nameI18nKey, descriptionI18nKey);
        permissionModulesPerPluginKey.put(plugin.getKey(), moduleHandle);

        // This should work now that we have dynamically created the module
        return getLicensingRole(plugin, key);
    }

    @Override
    public LicensingRole getLicensingRole(Plugin plugin, String key)
    {
        for (GlobalPermissionType type : permissionManager.getGlobalPermission(key))
        {
            final String pluginKey = plugin.getKey();
            final String typeKey = type.getKey();
            pluginKeysPerPermission.putIfAbsent(typeKey, new ConcurrentHashMap<String, String>());
            pluginKeysPerPermission.get(typeKey).putIfAbsent(pluginKey, pluginKey);

            return new JiraLicensingRole(permissionManager, userManager, type, globalPermissionUserCountCache);
        }

        return null;
    }

    @Override
    public void onPluginUnlicensedEvent(Plugin plugin)
    {
        for (ModuleRegistrationHandle handle : option(permissionModulesPerPluginKey.get(plugin.getKey())))
        {
            // remove the global permission plugin module such that it won't show up on the Global Permissions page
            handle.unregister();
            permissionModulesPerPluginKey.remove(plugin.getKey());
        }
    }

    /**
     * Handles the event that a global permission gets modified (e.g. a group gets added or removed from it).
     */
    @SuppressWarnings({"UnusedDeclaration"})
    @EventListener
    public void onGlobalPermissionEvent(AbstractGlobalPermissionEvent event)
    {
        handleEventForGlobalPermission(event.getGlobalPermissionType().getGlobalPermissionKey());
    }

    /**
     * Handles the event when something changes in the Crowd directory.
     */
    @SuppressWarnings({"UnusedDeclaration"})
    @EventListener
    public void onCrowdDirectoryEvent(DirectoryEvent event)
    {
        if (event instanceof GroupMembershipCreatedEvent)
        {
            // A user got added to a group. Only refresh the caches associated with global permissions relating to this group.
            handleEventForGroupName(((GroupMembershipCreatedEvent)event).getGroupName());
        }
        else if (event instanceof GroupMembershipDeletedEvent)
        {
            // A user got removed from a group. Only refresh the caches associated with global permissions relating to this group.
            handleEventForGroupName(((GroupMembershipDeletedEvent)event).getGroupName());
        }
        else
        {
            // We're not sure what particular group it was associated with, but we know that this event
            // may have implications on the cache. Although this is rather inefficient, clear all role-based caches.
            handleEventForUnidentifiedGlobalPermission();
        }
    }

    /**
     * Handles the event that a user gets removed from a group.
     */
    @SuppressWarnings({"UnusedDeclaration"})
    @EventListener
    public void onGroupMembershipDeletedEvent(GroupMembershipDeletedEvent event)
    {
        handleEventForGroupName(event.getGroupName());
    }

    /**
     * Handles the event that groups get synchronized with a remote directory
     */
    @SuppressWarnings({"UnusedDeclaration"})
    @EventListener
    public void onDirectorySynchronisation(RemoteDirectorySynchronisedEvent event)
    {
        handleEventForUnidentifiedGlobalPermission();
    }

    /**
     * Handles the event requesting that all of JIRA's caches be flushed
     */
    @SuppressWarnings({"UnusedDeclaration"})
    @EventListener
    public void onClearCache(ClearCacheEvent event)
    {
        handleEventForUnidentifiedGlobalPermission();
    }

    private void handleEventForGroupName(String groupName)
    {
        for (GlobalPermissionKey permission : getPermissionsWithGroup(groupName))
        {
            // UPM-4444 If a user was removed from the "Use JIRA" permission, we need to refresh all caches.
            if (GlobalPermissionKey.USE.equals(permission))
            {
                handleEventForUnidentifiedGlobalPermission();
            }
            else
            {
                handleEventForGlobalPermission(permission);
            }
        }
    }

    private void handleEventForGlobalPermission(GlobalPermissionKey permission)
    {
        globalPermissionUserCountCache.reset(permission);
        for (String pluginKey : getPluginKeysWithPermission(permission))
        {
            eventPublisher.publish(new LicensingRoleMembershipUpdatedEvent(pluginKey));
        }
    }

    private void handleEventForUnidentifiedGlobalPermission()
    {
        globalPermissionUserCountCache.resetAll();
        eventPublisher.publish(new LicensingRoleMembershipUpdatedEvent("*"));
    }

    @Override
    public void afterPropertiesSet() throws Exception
    {
        eventPublisher.register(this);
    }

    @Override
    public void destroy() throws Exception
    {
        eventPublisher.unregister(this);
    }

    private Iterable<String> getPluginKeysWithPermission(GlobalPermissionKey permissionKey)
    {
        Map<String, String> keysMap = pluginKeysPerPermission.get(permissionKey.getKey());
        return (keysMap == null) ? ImmutableList.<String>of() : ImmutableList.copyOf(keysMap.values());
    }

    private Iterable<GlobalPermissionKey> getPermissionsWithGroup(String groupName)
    {
        ImmutableList.Builder<GlobalPermissionKey> withGroup = ImmutableList.builder();

        for (GlobalPermissionType permission : permissionManager.getAllGlobalPermissions())
        {
            if (permissionManager.getGroupNamesWithPermission(permission.getGlobalPermissionKey()).contains(groupName))
            {
                withGroup.add(permission.getGlobalPermissionKey());
            }
        }

        return withGroup.build();
    }
}
