package com.atlassian.marketplace.client.impl;

import java.net.URI;
import java.util.Date;

import com.atlassian.marketplace.client.MpacException;
import com.atlassian.marketplace.client.api.ApplicationDetailQuery;
import com.atlassian.marketplace.client.api.ApplicationKey;
import com.atlassian.marketplace.client.api.Applications;
import com.atlassian.marketplace.client.impl.representations.ApplicationsRepresentation;
import com.atlassian.marketplace.client.impl.representations.RootRepresentation;
import com.atlassian.marketplace.client.model.Application;
import com.atlassian.marketplace.client.model.ApplicationSummary;
import com.atlassian.marketplace.client.model.ApplicationVersion;
import com.atlassian.marketplace.client.util.UriBuilder;
import com.atlassian.upm.api.util.Option;

import org.codehaus.jackson.annotate.JsonProperty;

import static com.atlassian.marketplace.client.impl.representations.RepresentationLinks.APPLICATIONS_REL;

final class ApplicationsImpl implements Applications
{
    private final DefaultMarketplaceClient client;
    private final RootRepresentation root;

    ApplicationsImpl(DefaultMarketplaceClient client, RootRepresentation root)
    {
        this.client = client;
        this.root = root;
    }

    @Override
    public Iterable<ApplicationSummary> findAll() throws MpacException
    {
        ApplicationsRepresentation appsRep = client.getEntity(applicationsBaseUri().build(), ApplicationsRepresentation.class);
        return appsRep.getApplications();
    }

    @Override
    public Option<Application> get(ApplicationDetailQuery query) throws MpacException
    {
        final UriBuilder uri = applicationsBaseUri().path(client.urlEncode(query.getAppKey().getKey()));
        for (ApplicationDetailQuery.BuildNumber build: query.getBuildNumber())
        {
            uri.queryParam(build.isGreaterThan() ? "build-number-after" : "build-number", build.getBuildNumber());
        }
        for (Integer limit: query.getLimitVersions())
        {
            uri.queryParam("limit-versions", limit);
        }
        return client.getOptionalEntity(uri.build(), Application.class);
    }

    @Override
    public Option<ApplicationVersion> getVersion(ApplicationKey application, long buildNumber) throws MpacException
    {
        final UriBuilder uri = applicationsBaseUri().path(client.urlEncode(application.getKey())).path(String.valueOf(buildNumber));
        return client.getOptionalEntity(uri.build(), ApplicationVersion.class);
    }

    @Override
    public void putVersion(ApplicationKey app, ApplicationVersion version) throws MpacException
    {
        URI uri = applicationsBaseUri().path(client.urlEncode(app.getKey()))
            .path(String.valueOf(version.getBuildNumber())).build();
        client.putEntity(uri, new ApplicationVersionPutData(version));
    }
    
    private UriBuilder applicationsBaseUri() throws MpacException
    {
        return UriBuilder.fromUri(client.requireLinkUri(root.getLinks(), APPLICATIONS_REL, root.getClass()));
    }
    
    static final class ApplicationVersionPutData
    {
        @JsonProperty final String version;
        @JsonProperty final Date releaseDate;
        @JsonProperty final String status;
        
        ApplicationVersionPutData(ApplicationVersion from)
        {
            this.version = from.getVersion();
            this.releaseDate = from.getReleaseDate();
            this.status = from.isPublished() ? "Published" : "Draft";
        }
    }
}
