package com.atlassian.templaterenderer.plugins;

import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.PluginParseException;
import com.atlassian.plugin.StateAware;
import com.atlassian.plugin.descriptors.AbstractModuleDescriptor;
import com.atlassian.plugin.module.ContainerManagedPlugin;
import com.atlassian.plugin.module.ModuleFactory;
import com.atlassian.plugin.remotable.api.annotation.ComponentImport;
import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import org.dom4j.Attribute;
import org.dom4j.Element;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.inject.Inject;

/**
 * Module descriptor for template context items.  These may either be a class that gets instantiated on each lookup, or
 * they may reference a component using component-ref.
 */
public class TemplateContextItemModuleDescriptor extends AbstractModuleDescriptor<Object> implements
    StateAware
{
    private Logger log = LoggerFactory.getLogger(TemplateContextItemModuleDescriptor.class);
    private boolean global = false;
    private String contextKey;
    private Supplier<Object> contextItem = null;
    private final BundleContext bundleContext;

    @Inject
    public TemplateContextItemModuleDescriptor(ModuleFactory moduleFactory,
            BundleContext bundleContext)
    {
        super(moduleFactory);
        this.bundleContext = bundleContext;
    }

    @Override public void init(Plugin plugin, Element element) throws PluginParseException
    {
        super.init(plugin, element);
        Attribute globalAtt = element.attribute("global");
        if (globalAtt != null)
        {
            global = Boolean.parseBoolean(globalAtt.getValue());
        }
        Attribute contextKeyAtt = element.attribute("context-key");
        if (contextKeyAtt == null)
        {
            throw new PluginParseException("context-key must be specified");
        }
        contextKey = contextKeyAtt.getValue();
        final Attribute componentImportRefAttr = element.attribute("component-import");
        final Attribute componentRefAttr = element.attribute("component-ref");
        Attribute classAttr = element.attribute("class");
        if (componentImportRefAttr != null)
        {
            contextItem = new Supplier<Object>()
            {
                public Object get()
                {
                    ServiceReference ref = bundleContext.getServiceReference(componentImportRefAttr.getValue());
                    return ref != null ? bundleContext.getService (ref) : null;
                }
            };
        }
        else if (componentRefAttr != null)
        {
            contextItem = new Supplier<Object>()
            {
                public Object get()
                {
                    // We don't cache componentRefs, because if the user wants to use a prototype component, then
                    // caching it would undermine that.  It's just a hash map lookup anyway.
                    if (getPlugin() instanceof ContainerManagedPlugin)
                    {
                        return ((ContainerManagedPlugin)getPlugin()).getContainerAccessor().getBean(componentRefAttr.getValue());
                    }
                    throw new IllegalStateException("Plugin isn't a container-managed plugin");
                }
            };
        }
        else if (classAttr != null)
        {
            contextItem = Suppliers.memoize(new Supplier<Object>()
            {
                public Object get()
                {
                    return moduleFactory.createModule(getModuleClassName(), TemplateContextItemModuleDescriptor.this);
                }
            });
        }
        else
        {
            throw new PluginParseException("You must specify a class or a component-ref");
        }
    }

    @Override
    public synchronized Object getModule()
    {
        return contextItem.get();
    }

    @Override public synchronized void disabled()
    {
        super.disabled();
        contextItem = null;
    }

    public boolean isGlobal()
    {
        return global;
    }

    public String getContextKey()
    {
        return contextKey;
    }
}
