package com.atlassian.support.tools.salext.bundle;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;

import org.apache.log4j.Logger;

import com.atlassian.support.tools.salext.SupportApplicationInfo;

public class CustomisationFileBundle extends AbstractApplicationFileBundle
{
    private static final Logger log = Logger.getLogger(CustomisationFileBundle.class);
    
    private final TreeMap<String, String> customDecorators;
    private final HashMap<String, String> customHtml;
    private final HashMap<String, String> customStylesheet;
    private final SupportApplicationInfo info;
    private final String LAYOUTS = "layouts";
    
    private final Map<String, String> customisedFiles = new HashMap<String, String>();
    
    public CustomisationFileBundle(BundleManifest bundle, String title, String description, SupportApplicationInfo info, 
                    TreeMap<String, String> customDecorators, HashMap<String, String> customHtml, HashMap<String, String> customStylesheet)
    {
        super(bundle, title, description);
        this.customDecorators = customDecorators;
        this.customHtml = customHtml;
        this.customStylesheet = customStylesheet;
        this.info = info;
    }

    @Override
    public Map<String, String> getFiles()
    {
        // Create export directory
        final File supportDir = new File(info.getApplicationHome(), "logs/support");
        
        if(supportDir.exists() || supportDir.mkdirs())
        {
            customisedFiles.putAll(getCustomLayouts(supportDir));
            customisedFiles.putAll(getCustomHtml(supportDir));
            customisedFiles.putAll(getCustomStylesheet(supportDir));
        }
        else 
        {
        	log.error("Couldn't create export directory " + supportDir.getAbsolutePath());
        }
        
        return customisedFiles;
    }

    /**
     * Return a map containing the filename and an optional subfolder to put that file in in the zip  
     * @param directory the directory to create the temp file in
     * @return the filename and an optional subfolder for the zip
     */
	private Map<String, String> getCustomLayouts(File directory)
	{
		final Map<String, String> tmpCustomLayouts = new TreeMap<String, String>();
		
        // Process all custom layouts
        final File customisedLayoutsFile = new File(directory,"customLayouts.txt");
        try
        {
            final FileWriter layoutsWriter = new FileWriter(customisedLayoutsFile);
            try
            {
                final StringBuilder customisationsString = new StringBuilder();
                
                if (customDecorators.isEmpty())
                {
                        customisationsString.append("No custom layouts");
                }
                
                for (String file : customDecorators.keySet())
                {
                    // Add to the list of customised files
                    customisationsString.append(file + "\n");
            
                    try
                    {
                        // Process the file itself
                        final File vmdFile = new File(directory, file);
                        
                        if(vmdFile.getParentFile().exists() || vmdFile.getParentFile().mkdirs())
                        {
                        	final FileWriter vmdWriter = new FileWriter(vmdFile);
	                        
	                        try
	                        {
	                            vmdWriter.write(customDecorators.get(file));
	                        }
	                        catch(IOException e)
	                        {
	                            log.error("Failed to write vmd to " + vmdFile.getPath() + ".", e);
	                        }
	                        finally
	                        {
	                            vmdWriter.flush();
	                            vmdWriter.close();
	                        }
	                
	                        // Add the vmd file
	                        tmpCustomLayouts.put(vmdFile.getAbsolutePath(), LAYOUTS);
                        }
                        else
                        {
                        	log.error("Couldn't create vmd directory " + vmdFile.getParentFile().getAbsolutePath());
                        }
                    }
                    catch(Exception e)
                    {
                        log.error("Can't generate vmd file.", e);
                    }
                }
                
                layoutsWriter.write(customisationsString.toString());
            }
            catch(IOException e)
            {
                log.error("Failed to write customisations to "+customisedLayoutsFile.getPath()+".", e);
            }
            finally
            {
                layoutsWriter.flush();
                layoutsWriter.close();
            }
            
            // Add the list of custom layouts
            tmpCustomLayouts.put(customisedLayoutsFile.getAbsolutePath(), LAYOUTS);
        }
        catch(Exception e)
        {
            log.error("Can't generate customised layouts file.", e);
        }

        return tmpCustomLayouts;
	}
	
    /**
     * Return a map containing the filename and an optional subfolder to put that file in in the zip  
     * @param directory the directory to create the temp file in
     * @return the filename and an optional subfolder for the zip
     */
	private Map<String, String> getCustomHtml(File directory)
	{
		final Map<String, String> tmpCustomHtml = new HashMap<String, String>();
		
        try
        {
	        // Process the custom HTML
			final File customisedHtmlFile = new File(directory,"customHtml.txt");
	        final FileWriter htmlWriter = new FileWriter(customisedHtmlFile);
	     
	        try
	        {
                final StringBuilder htmlString = new StringBuilder();
                
                if (customHtml.isEmpty())
                {
                    htmlString.append("No custom HTML");
                }
                
                for (String htmlObj : customHtml.keySet())
                {
                    // Add to the html to the file
                    htmlString.append(htmlObj + "\n" + customHtml.get(htmlObj) + "\n\n");
                }
                
                htmlWriter.write(htmlString.toString());
	        }
	        catch(IOException e)
	        {
                log.error("Failed to write customisations to "+customisedHtmlFile.getPath()+".", e);
	        }
	        finally
	        {
                htmlWriter.flush();
                htmlWriter.close();
	        }
	        
	        // Add the custom html
	        tmpCustomHtml.put(customisedHtmlFile.getAbsolutePath(), "");
        }
        catch(Exception e)
        {
            log.error("Can't generate custom html file.", e);
        }
        
        return tmpCustomHtml;
	}
	
    /**
     * Return a map containing the filename and an optional subfolder to put that file in in the zip  
     * @param directory the directory to create the temp file in
     * @return the filename and an optional subfolder for the zip
     */
	private Map<String, String> getCustomStylesheet(File directory)
	{
		final Map<String, String> tmpCustomStylesheet = new HashMap<String, String>();
        
        try
        {
			// Process the custom stylesheet
	        final File customStylesheetFile = new File(directory,"customStylesheet.txt");
	        final FileWriter stylesheetWriter = new FileWriter(customStylesheetFile);
	        try
	        {
                final StringBuilder stylesheetString = new StringBuilder();
                
                if (customStylesheet.isEmpty())
                {
                    stylesheetString.append("No custom stylesheet");
                }
                
                for (String stylesheetObj : customStylesheet.keySet())
                {
                    // Add to the html to the file
                    stylesheetString.append(stylesheetObj + "\n" + customStylesheet.get(stylesheetObj) + "\n\n");
                }
                
                stylesheetWriter.write(stylesheetString.toString());
	        }
	        catch(IOException e)
	        {
                log.error("Failed to write customisations to "+customStylesheetFile.getPath()+".", e);
	        }
	        finally
	        {
                stylesheetWriter.flush();
                stylesheetWriter.close();
	        }
	        
	        // Add the custom stylesheet
	        tmpCustomStylesheet.put(customStylesheetFile.getAbsolutePath(), "");
        }
        catch(Exception e)
        {
            log.error("Can't generate custom stylesheet file.", e);
        }
        
        return tmpCustomStylesheet;
	}
}