package com.atlassian.support.tools.properties;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.List;
import java.util.Map;

/**
 * Hierarchical store for support information. {@code PropertyStore} contains key-value pairs and can contain nested
 * {@code PropertyStore} nodes. These nodes are called "categories".
 * </p>
 * Nested {@code PropertyStore} instances can be created through the {@link #addCategory(String)} and
 * {@link #addCategory(String, PropertyStore)} methods. {@code PropertyStore} can contain multiple nested
 * {@code PropertyStore} instances for a category key.
 */
public interface PropertyStore {
    /**
     * Creates a new nested {@code PropertyStore} and registers it under the provided {@code categoryKey}.
     *
     * @param categoryKey the key of the nested {@code PropertyStore}.
     * @return the newly created nested {@link PropertyStore}
     */
    @Nonnull
    PropertyStore addCategory(@Nonnull String categoryKey);

    /**
     * Registers the provided {@code store} under the provided {@code categoryKey}.
     *
     * @param categoryKey the key of the nested {@code PropertyStore}
     * @param store the {@code PropertyStore}
     * @return the newly registered {@link PropertyStore}
     */
    @Nonnull
    PropertyStore addCategory(@Nonnull String categoryKey, @Nonnull PropertyStore store);

    /**
     * @return the map of nested categories. The map is a multi-map; multiple {@code PropertyStore} instances can be
     *         created/registered under a category key.
     */
    @Nonnull
    Map<String, List<PropertyStore>> getCategories();

    /**
     * @return the properties
     */
    @Nonnull
    Map<String, String> getValues();

    /**
     * Adds all provided properties to the {@code PropertyStore}. Note that calling this method only <em>adds</em> the
     * provided provided properties. It does not <em>replace</em> the current properties with the provided ones.
     *
     * @param values the properties.
     */
    void putValues(@Nonnull Map<String, String> values);

    /**
     * Updates a property.
     *
     * @param name the property name
     * @param value the new value, can be {@code null}
     */
    void setValue(@Nonnull String name, @Nullable String value);
}
