package com.atlassian.support.tools.properties;

import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.event.PluginEventManager;
import com.atlassian.plugin.tracker.DefaultPluginModuleTracker;
import com.atlassian.plugin.tracker.PluginModuleTracker;
import com.atlassian.support.tools.spi.SupportInfoAppender;
import com.atlassian.support.tools.spi.SupportInfoBuilder;
import com.atlassian.support.tools.spi.SupportInfoModuleDescriptor;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.DisposableBean;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import static com.google.common.base.Preconditions.*;
import static com.google.common.collect.Iterables.filter;
import static com.google.common.collect.Lists.newArrayList;

public class PluginSupportInfoAppenderManager implements SupportInfoAppenderManager, DisposableBean
{
    private static final Logger log = Logger.getLogger(PluginSupportInfoAppenderManager.class);

    private final PluginModuleTracker<SupportInfoAppender<?>, SupportInfoModuleDescriptor> moduleTracker;
    private final int maximumContextObjectDepth;
    private final int maximumCategoryDepth;

    public PluginSupportInfoAppenderManager(PluginAccessor pluginAccessor, PluginEventManager pluginEventManager)
    {
        this(new DefaultPluginModuleTracker<SupportInfoAppender<?>, SupportInfoModuleDescriptor>(pluginAccessor,
                pluginEventManager, SupportInfoModuleDescriptor.class));
    }

    protected PluginSupportInfoAppenderManager(PluginModuleTracker<SupportInfoAppender<?>, SupportInfoModuleDescriptor> moduleTracker)
    {
        maximumContextObjectDepth = Integer.valueOf(System.getProperty("stp.spi.contexts.maximum", "1000"));
        maximumCategoryDepth = Integer.valueOf(System.getProperty("stp.spi.categories.maximum", "25"));
        this.moduleTracker = checkNotNull(moduleTracker);
    }

    @Override
    public void destroy()
    {
        moduleTracker.close();
    }

    @Override
    public void addSupportInfo(PropertyStore propertyStore)
    {
        checkNotNull(propertyStore);
        SupportInfoBuilder builder = new PropertyStoreSupportInfoBuilder(propertyStore);
        addSupportInfo(builder, null);
    }

    private <T> void addSupportInfo(SupportInfoBuilder builder, T context)
    {
        List<SupportInfoModuleDescriptor> filteredModuleDescriptors = newArrayList(filter(moduleTracker.getModuleDescriptors(), contextClassMatches(context)));
        Collections.sort(filteredModuleDescriptors);
        for (SupportInfoModuleDescriptor descriptor : filteredModuleDescriptors)
        {
            //noinspection unchecked
            SupportInfoAppender<T> appender = (SupportInfoAppender<T>) descriptor.getModule();
            appender.addSupportInfo(builder, context);
        }
    }

    private static Predicate<SupportInfoModuleDescriptor> contextClassMatches(final Object context)
    {
        return new Predicate<SupportInfoModuleDescriptor>()
        {
            @Override
            public boolean apply(SupportInfoModuleDescriptor descriptor)
            {
                Class<?> aClass = Void.class;
                if (context != null)
                {
                    aClass = context.getClass();
                }
                return descriptor.getContextClass().isAssignableFrom(aClass);
            }
        };
    }

    private class PropertyStoreSupportInfoBuilder implements SupportInfoBuilder
    {
        private final int countOfContextObjects;
        private final PropertyStore propertyStore;
        private final Set<Integer> allContextObjectsIds;
        private int currentCategoryDepth;

        private PropertyStoreSupportInfoBuilder(PropertyStore propertyStore)
        {
            this.propertyStore = checkNotNull(propertyStore);
            this.allContextObjectsIds = new HashSet<Integer>();
            this.countOfContextObjects = 0;
            this.currentCategoryDepth = 0;
        }

        private PropertyStoreSupportInfoBuilder(PropertyStoreSupportInfoBuilder parentBuilder, PropertyStore propertyStore)
        {
            this.propertyStore = checkNotNull(propertyStore);
            this.allContextObjectsIds = parentBuilder.allContextObjectsIds;
            this.countOfContextObjects = parentBuilder.countOfContextObjects;
            this.currentCategoryDepth = parentBuilder.currentCategoryDepth + 1;
        }

        private PropertyStoreSupportInfoBuilder(PropertyStoreSupportInfoBuilder parentBuilder, Object context)
        {
            this.propertyStore = parentBuilder.propertyStore;
            this.allContextObjectsIds = ImmutableSet.<Integer>builder().addAll(parentBuilder.allContextObjectsIds).add(System.identityHashCode(context)).build();
            this.countOfContextObjects = parentBuilder.countOfContextObjects + 1;
            this.currentCategoryDepth = parentBuilder.currentCategoryDepth;
        }

        @Override
        public SupportInfoBuilder addValue(String key, String value)
        {
            checkArgument(StringUtils.isNotBlank(key), "empty or blank key");
            if (StringUtils.isNotEmpty(value))
            {
                propertyStore.setValue(key, value);
            }
            return this;
        }

        @Override
        public SupportInfoBuilder addCategory(String categoryKey)
        {
            //this should be found at coding time so it is an exception
            checkState(!(currentCategoryDepth > maximumCategoryDepth), "Maximum number of categories reached");
            checkNotNull(categoryKey, "categoryKey");
            return new PropertyStoreSupportInfoBuilder(this, propertyStore.addCategory(categoryKey));
        }

        @Override
        public <T> SupportInfoBuilder addContext(T context)
        {
            checkState(!allContextObjectsIds.contains(System.identityHashCode(context)), "Cannot have recursive context objects");
            checkNotNull(context, "context");
            PropertyStoreSupportInfoBuilder child = new PropertyStoreSupportInfoBuilder(this, context);
            if (countOfContextObjects < maximumContextObjectDepth)
            {
                addSupportInfo(child, context);
            } else
            {
                //this is more likely to occur for large instances so not an exception (we still want support info and not a bang)
                log.warn("Maximum number of context objects reached, not calling child appenders");
            }
            return child;
        }
    }

}
