package com.atlassian.xwork.interceptors;

import com.atlassian.annotations.security.XsrfProtectionExcluded;
import com.atlassian.xwork.HttpMethod;
import com.atlassian.xwork.RequireSecurityToken;
import com.atlassian.xwork.SimpleXsrfTokenGenerator;
import com.atlassian.xwork.XsrfTokenGenerator;
import com.opensymphony.xwork2.Action;
import com.opensymphony.xwork2.ActionInvocation;
import com.opensymphony.xwork2.ActionSupport;
import com.opensymphony.xwork2.interceptor.Interceptor;
import com.opensymphony.xwork2.interceptor.ValidationAware;
import org.apache.struts2.ServletActionContext;

import javax.servlet.http.HttpServletRequest;
import java.lang.reflect.Method;

/**
 * Interceptor to add XSRF token protection to XWork actions.
 * <p>
 * Configuring XSRF protection happens at the method
 * level, and can be done either by adding a {@link XsrfProtectionExcluded} or {@link RequireSecurityToken} annotation
 * to the method, or by adding a
 * &lt;param name="RequireSecurityToken"&gt;[true|false]&lt;/param&gt; parameter to the action configuration in
 * <code>xwork.xml</code>.
 * <p>
 * {@code XsrfProtectionExcluded} annotations override any other settings. Configuration in xwork.xml will override
 * {@code RequireSecurityToken} annotations. Behaviour when a method is not configured at all depends on the return
 * values of {@link #getSecurityLevel()}.
 * <p>
 * Requests containing the HTTP header <code>X-Atlassian-Token: no-check</code> will bypass the check and always
 * succeed.
 *
 * @see SecurityLevel
 * @see #getSecurityLevel()
 */
public class XsrfTokenInterceptor implements Interceptor {
    public static final String REQUEST_PARAM_NAME = "atl_token";
    public static final String CONFIG_PARAM_NAME = "RequireSecurityToken";
    public static final String VALIDATION_FAILED_ERROR_KEY = "atlassian.xwork.xsrf.badtoken";
    public static final String SECURITY_TOKEN_REQUIRED_ERROR_KEY = "atlassian.xwork.xsrf.notoken";
    public static final String OVERRIDE_HEADER_NAME = "X-Atlassian-Token";
    public static final String OVERRIDE_HEADER_VALUE = "no-check";

    public enum SecurityLevel {
        /**
         * Methods without any configuration are not protected by default.
         *
         * @deprecated since 2.1. Use {@link #OPT_OUT} or {@link #DEFAULT} instead.
         */
        @Deprecated
        OPT_IN {
            @Override
            public boolean getDefaultProtection() {
                return false;
            }
        },
        /**
         * Methods without any configuration are protected by default.
         */
        OPT_OUT {
            @Override
            public boolean getDefaultProtection() {
                return true;
            }
        },
        /**
         * HTTP methods classified as "safe" and "idempotent" according to RFC 7231 are not protected by
         * default, all other methods are protected by default.
         * <p>
         * Safe and idempotent HTTP methods are GET, HEAD, OPTIONS and TRACE.
         *
         * @since 2.1
         */
        DEFAULT {
            @Override
            public boolean getDefaultProtection() {
                String httpMethod = getHttpMethod();
                return !HttpMethod.anyMatch(httpMethod,
                        HttpMethod.GET, HttpMethod.HEAD, HttpMethod.OPTIONS, HttpMethod.TRACE);
            }
        };

        public abstract boolean getDefaultProtection();
    }

    private final XsrfTokenGenerator tokenGenerator;

    public XsrfTokenInterceptor() {
        this(new SimpleXsrfTokenGenerator());
    }

    public XsrfTokenInterceptor(XsrfTokenGenerator tokenGenerator) {
        this.tokenGenerator = tokenGenerator;
    }

    public String intercept(ActionInvocation invocation) throws Exception {
        Method invocationMethod = extractMethod(invocation);
        String configParam = (String) invocation.getProxy().getConfig().getParams().get(CONFIG_PARAM_NAME);
        RequireSecurityToken legacyAnnotation = invocationMethod.getAnnotation(RequireSecurityToken.class);
        XsrfProtectionExcluded annotation = invocationMethod.getAnnotation(XsrfProtectionExcluded.class);

        boolean isProtected = methodRequiresProtection(configParam, annotation, legacyAnnotation);
        String token = ServletActionContext.getRequest().getParameter(REQUEST_PARAM_NAME);
        boolean validToken = tokenGenerator.validateToken(ServletActionContext.getRequest(), token);

        if (isProtected && !validToken) {
            Action action = (Action) invocation.getAction();
            String errorMessageKey = (token == null
                    ? SECURITY_TOKEN_REQUIRED_ERROR_KEY
                    : VALIDATION_FAILED_ERROR_KEY);
            addInvalidTokenError(action, errorMessageKey);
            ServletActionContext.getResponse().setStatus(403);
            return ActionSupport.INPUT;
        }

        return invocation.invoke();
    }

    private static String getHttpMethod() {
        HttpServletRequest servletRequest = ServletActionContext.getRequest();
        return servletRequest == null ? "" : servletRequest.getMethod();
    }

    private static Method extractMethod(ActionInvocation invocation) throws NoSuchMethodException {
        final Class<?> actionClass = invocation.getAction().getClass();
        final String methodName = invocation.getProxy().getMethod();
        return actionClass.getMethod(methodName);
    }

    private boolean methodRequiresProtection(
            String configParam,
            XsrfProtectionExcluded annotation,
            RequireSecurityToken legacyAnnotation) {
        if (isOverrideHeaderPresent()) {
            return false;
        }
        if (annotation != null) {
            return false;
        }
        if (configParam != null) {
            return Boolean.parseBoolean(configParam);
        }
        if (legacyAnnotation != null) {
            return legacyAnnotation.value();
        }
        return getSecurityLevel().getDefaultProtection();
    }

    /**
     * Add error to action in cases where token is required, but is missing or invalid. Implementations may
     * wish to override this method, but most should be able to get away with just overriding
     * {@link #internationaliseErrorMessage}
     *
     * @param action          the action to add the error message to
     * @param errorMessageKey the error message key that will be used to internationalise the message
     */
    protected void addInvalidTokenError(Action action, String errorMessageKey) {
        if (action instanceof ValidationAware)
            ((ValidationAware) action).addActionError(internationaliseErrorMessage(action, errorMessageKey));
    }

    /**
     * Convert an error message key into the correct message for the current user's locale. The default implementation
     * is only useful for testing. Implementations should override this method to provide the appropriate
     * internationalised implementation.
     *
     * @param action     the current action being executed
     * @param messageKey the message key that needs internationalising
     * @return the appropriate internationalised message for the current user
     */
    protected String internationaliseErrorMessage(Action action, String messageKey) {
        return messageKey;
    }

    private boolean isOverrideHeaderPresent() {
        return OVERRIDE_HEADER_VALUE.equals(ServletActionContext.getRequest().getHeader(OVERRIDE_HEADER_NAME));
    }

    ///CLOVER:OFF

    public void destroy() {
    }

    public void init() {
    }

    /**
     * Gets the current security level. See {@link SecurityLevel} for more information on the meanings of the different
     * level. Default implementation returns {@link SecurityLevel#DEFAULT}. Implementations should
     * override this method if they want more control over the security level setting.
     *
     * @return the security level to apply to this interceptor.
     */
    protected SecurityLevel getSecurityLevel() {
        return SecurityLevel.DEFAULT;
    }
}
