package com.atlassian.streams.internal;

import com.atlassian.sal.api.ApplicationProperties;
import com.atlassian.sal.api.user.UserManager;
import com.atlassian.streams.api.common.Either;
import com.atlassian.streams.api.common.Pair;
import com.atlassian.streams.api.common.uri.Uri;
import com.atlassian.streams.internal.ActivityProvider.Error;
import com.atlassian.streams.internal.feed.FeedAggregator;
import com.atlassian.streams.internal.feed.FeedModel;
import com.atlassian.streams.internal.feed.builder.ToFeedCallable;
import com.atlassian.streams.spi.CancellableTask;
import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.net.URI;
import java.util.Collections;
import java.util.Set;

import static com.atlassian.streams.api.common.Either.left;
import static com.atlassian.streams.api.common.Pair.pair;
import static com.atlassian.streams.internal.ActivityProvider.Error.other;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Predicates.in;
import static com.google.common.base.Predicates.not;
import static com.google.common.collect.Iterables.concat;
import static com.google.common.collect.Iterables.filter;
import static com.google.common.collect.Iterables.transform;
import static java.util.concurrent.TimeUnit.MILLISECONDS;

public final class FeedBuilder
{
    private final Logger log = LoggerFactory.getLogger(FeedBuilder.class);

    private final ActivityProviders activityProviders;
    private final FeedAggregator aggregator;
    private final StreamsCompletionService completionService;
    private final ApplicationProperties applicationProperties;
    private final UserManager userManager;

    public FeedBuilder(ActivityProviders activityProviders,
                       FeedAggregator aggregator,
                       StreamsCompletionService completionService,
                       ApplicationProperties applicationProperties,
                       UserManager userManager)
    {
        this.applicationProperties = applicationProperties;
        this.activityProviders = checkNotNull(activityProviders, "activityProviders");
        this.aggregator = checkNotNull(aggregator, "aggregator");
        this.completionService = checkNotNull(completionService, "completionService");
        this.userManager = checkNotNull(userManager, "userManager");
    }

    public FeedModel getFeed(Uri self, String contextPath, HttpParameters parameters, String requestLanguages)
    {
        if(parameters.allowOnlyAuthorized() && userManager.getRemoteUserKey() == null)
        {
            Set<Either<Error, FeedModel>> empty = Collections.emptySet();
            return aggregator.aggregate(empty, self, 0, parameters.getTitle());
        }
        Iterable<ActivityProvider> providers = activityProviders.get(ImmutableSet.of(
                parameters.module(), parameters.fetchLocalOnly(), parameters.isSelectedProvider()));
        final ImmutableSet<ActivityProvider> banned = ImmutableSet.copyOf(filter(providers, not(completionService.reachable())));
        final Iterable<ActivityProvider> notBannedProviders = filter(providers, not(in(banned)));

        final Iterable<ActivityProviderCancellableTask<Either<Error, FeedModel>>> callables = transform(notBannedProviders,
                toFeedCallable(pair(self, parameters),
                        parameters.calculateContextUrl(applicationProperties, contextPath), requestLanguages));

        final Iterable<Either<Error, FeedModel>> results;
        if (Sys.inDevMode() && !parameters.isTimeoutTest())
        {
            results = completionService.execute(callables);
        }
        else
        {
            results = completionService.execute(callables, ActivityRequestImpl.DEFAULT_TIMEOUT, MILLISECONDS);
        }
        return aggregator.aggregate(concat(results, transform(banned, toLeftBanned())), self,
                parameters.parseMaxResults(ActivityRequestImpl.DEFAULT_MAX_RESULTS), parameters.getTitle());
    }

    private final Function<ActivityProvider, ActivityProviderCancellableTask<Either<Error, FeedModel>>> toFeedCallable(
            Pair<Uri, HttpParameters> feedParameters,
            URI baseUri,
            String requestLanguages)
    {
        return new ToFeedCallable(feedParameters, baseUri, requestLanguages);
    }

    private Function<ActivityProvider, Either<Error, FeedModel>> toLeftBanned()
    {
        return new Function<ActivityProvider, Either<Error, FeedModel>>()
        {
            @Override
            public Either<Error, FeedModel> apply(@Nullable final ActivityProvider activityProvider)
            {
                return Either.left(Error.banned(activityProvider));
            }
        };
    }

}
