define('widget/markup-editor', [
    'jquery',
    'underscore',
    'util/events',
    'widget/markup-attachments',
    'widget/markup-preview',
    'widget/mentionable-textarea'
], function(
    $,
    _,
    events,
    MarkupAttachments,
    MarkupPreview,
    MentionableTextarea
) {

    'use strict';

    /**
     * A top level component to simplify initialisation and destruction of markup editing sub-components
     *
     * @param {HTMLElement|jQuery} editor
     * @constructor MarkupEditor
     */
    function MarkupEditor(editor) {
        this.init.apply(this, arguments);
    }

    MarkupEditor._dataKey = 'markup-editor';

    /**
     * Convenience method for initialising a MarkupEditor and storing a reference to it in the jQuery data for the editor element
     *
     * @param {HTMLElement|jQuery} editor
     * @returns {MarkupEditor}
     */
    MarkupEditor.bindTo = function(editor){
        var $editor = MarkupEditor.unbindFrom(editor);

        var markupEditor = new MarkupEditor($editor);

        $editor.data(MarkupEditor._dataKey, markupEditor);

        return markupEditor;
    };

    /**
     * Destroy the MarkupEditor instance associated with the supplied editor element
     *
     * @param {HTMLElement|jQuery} editor
     * @returns {jQuery}
     */
    MarkupEditor.unbindFrom = function(editor){
        var $editor = $(editor);
            $editor = $editor.is('.markup-editor') ? $editor : $editor.find('.markup-editor');
        var markupEditor = $editor.data(MarkupEditor._dataKey);

        markupEditor && markupEditor.destroy();

        return $editor;
    };

    events.addLocalEventMixin(MarkupEditor.prototype);

    /**
     * Initialise the editor
     *
     * @param {HTMLElement|jQuery} editor
     */
    MarkupEditor.prototype.init = function(editor) {
        var $editor = $(editor);
        this.$editor = $editor.is('.markup-editor') ? $editor : $editor.find('.markup-editor');
        this._destroyables = [];
        var $textarea = this.$editor.find('textarea');

        if ($textarea.is(document.activeElement)) {
            //expandingTextarea removes $textarea from the DOM to insert it into its new container.
            //This preserves focus
            _.defer($textarea.focus.bind($textarea));
        }
        var $expandingTextarea = this.$editor.find('textarea.expanding').expandingTextarea();
        this._destroyables.push(events.chainWith($expandingTextarea).on('resize.expanding', this.onComponentResize.bind(this, false)));
        this._destroyables.push({ destroy: $expandingTextarea.expandingTextarea.bind($expandingTextarea, 'destroy') });

        var markupPreview = new MarkupPreview(this.$editor);
        this._destroyables.push(events.chainWith(markupPreview).on('resize', this.onComponentResize.bind(this, true)));
        this._destroyables.push(markupPreview);

        this._destroyables.push(new MentionableTextarea({$container: this.$editor}));


        var disableAttachmentsButton = function(message) {
            this.$editor.find('.markup-attachments-button')
                .addClass('disabled')
                .prop('disabled', 'disabled')
                .attr('aria-disabled', true)
                .attr('title', message);
        }.bind(this);

        if (MarkupAttachments.isEnabled()) {
            if (MarkupAttachments.isSupported()) {
                this._destroyables.push(new MarkupAttachments(this.$editor));
            } else {
                //IE9
                disableAttachmentsButton(AJS.I18n.getText('stash.web.markup.attachments.unsupported'));
            }
        } else {
            //Disabled on server
            disableAttachmentsButton(AJS.I18n.getText('stash.web.markup.attachments.disabled'));
        }
    };

    /**
     * Trigger an editor resize event whenever any of the components resize
     * @param {boolean} shouldUpdatePosition - A flag to say whether the component resize changes the focal point for the editor
     */
    MarkupEditor.prototype.onComponentResize = function(shouldUpdatePosition){
        this.$editor.triggerHandler('resize'); //Let any components listening for editor resizes know, but don't bubble
        this.trigger('resize', shouldUpdatePosition);
    };

    /**
     * Destroy the instance
     */
    MarkupEditor.prototype.destroy = function() {
        _.invoke(this._destroyables, 'destroy');

        this.$editor.data(MarkupEditor._dataKey, null);
        this.$editor = null;
        this.off();
    };

    return MarkupEditor;
});
