define('widget/keyboard-shortcuts', [
    'aui',
    'jquery',
    'underscore',
    'util/events',
    'util/navigator',
    'exports'
], function(
    AJS,
    $,
    _,
    events,
    navigatorUtil,
    exports) {

    'use strict';

    var CTRL = /^[cC]trl$/i,
        CMD = '\u2318'; // Mac `command` key symbol

    var _shortcutsByDisplayContext = {};

    function KeyboardShortcuts() {
        if (!(this instanceof KeyboardShortcuts)) {
            return new KeyboardShortcuts(trigger, registry);
        }

        this._dialog = new AJS.Dialog({
            width:830,
            height:580,
            id:"keyboard-shortcut-dialog",
            closeOnOutsideClick: true
        });

        this._enabledContexts = [];
    }

    KeyboardShortcuts.prototype._setRegistry = function(registry) {
        this._registry = registry;
    };

    KeyboardShortcuts.prototype._initContent = function() {
        this._dialog.addHeader(AJS.I18n.getText('stash.web.keyboardshortcut.header'));
        this._dialog.addPanel('', stash.widget.keyboardShortcutsContent({
            contextNames:_.keys(_shortcutsByDisplayContext),
            contexts:_.values(_shortcutsByDisplayContext)
        }));
        this._dialog.addCancel(AJS.I18n.getText('stash.web.button.close'), function (dialog) {
            dialog.hide();
        });
    };

    KeyboardShortcuts.prototype._bind = function($trigger) {
        this._$trigger = $trigger;
        var self = this;
        this._$trigger.on('click', function(e) {
              self._show();
              e.preventDefault();
        });
    };


    KeyboardShortcuts.prototype.enableContext = function(context) {
        if ($.inArray(context, this._enabledContexts) !== -1) {
            return;
        }
        this._registry.enableContext(context);
        this._enabledContexts.push(context);
    };

    KeyboardShortcuts.prototype.resetContexts = function() {
        AJS.trigger("remove-bindings.keyboardshortcuts");
        this._enabledContexts = [];
        AJS.trigger("add-bindings.keyboardshortcuts");
    };

    KeyboardShortcuts.prototype._show = function() {
        //If this is the first time shown, init the content
        if (!this._hasShown) {
            this._initContent();
            this._hasShown = true;
        }
        this._dialog.show();
        this._dialog.updateHeight();
        this._dialog.getCurrentPanel().body.find('.keyboard-shortcut-help').focus();
    };

    KeyboardShortcuts.prototype.addCustomShortcut = function(context, keys, description, displayContext) {
        var shortcut = internalizeShortcut({
            keys : keys,
            context : context,
            displayContext : displayContext,
            description : description
        }, { convertOSModifier : false });
    };

    KeyboardShortcuts.convertOSModifier = function(key) {
        return navigatorUtil.isMac() ? key.replace(CTRL, CMD) : key;
    };

    function internalizeShortcut(shortcut, options) {
        //need to do a copy to avoid messing up the shortcuts for whenIType
        shortcut = $.extend({}, shortcut);
        shortcut.keys = _.map(shortcut.keys, function(option) {
            return _.map(option, function(keypress) {
                if (_.all(['key', 'modifiers'], _.partial(_.has, keypress))) {
                    return keypress;
                }

                //Don't split on '+' when keypress length is 1, in case keypress is '+' only.
                var presses = (keypress.length > 1) ? keypress.split("+") : keypress;
                if (!_.isArray(presses) || presses.length === 1) {
                    return keypress;
                }
                return {
                    'key': presses.pop(),
                    // default is to convert the modifier
                    'modifiers': options && options.convertOSModifier === false ?
                                 presses :
                                 _.map(presses, KeyboardShortcuts.convertOSModifier)
                };
            });
        });

        if (!shortcut.displayContext) {
            shortcut.displayContext = KeyboardShortcuts._contextDisplayInfo[shortcut.context] ?
                KeyboardShortcuts._contextDisplayInfo[shortcut.context].displayName :
                shortcut.context.replace(/\b[a-z]/g, function(str) {
                    return str.toUpperCase();
                });
        }

        if (!_shortcutsByDisplayContext[shortcut.displayContext]) {
            _shortcutsByDisplayContext[shortcut.displayContext] = [];
        }
        _shortcutsByDisplayContext[shortcut.displayContext].push(shortcut);
    }

    KeyboardShortcuts.internalizeShortcuts = function(shortcuts) {
        _.each(shortcuts, internalizeShortcut);
    };

    KeyboardShortcuts._contextDisplayInfo = {
        'repository' : { displayName : AJS.I18n.getText('stash.web.keyboardshortcut.context.repository') },
        'branch-compare' : { displayName : AJS.I18n.getText('stash.web.keyboardshortcut.context.branch-compare') },
        'branch-list' : { displayName : AJS.I18n.getText('stash.web.keyboardshortcut.context.branch-list') },
        'changeset' : { displayName : AJS.I18n.getText('stash.web.keyboardshortcut.context.changeset') },
        'commits' : { displayName : AJS.I18n.getText('stash.web.keyboardshortcut.context.commits') },
        'diff-tree' : { displayName : AJS.I18n.getText('stash.web.keyboardshortcut.context.diff-tree') }, //Map this to changeset too
        'diff-view' : { displayName : AJS.I18n.getText('stash.web.keyboardshortcut.context.diff-view') },
        'filebrowser' : { displayName : AJS.I18n.getText('stash.web.keyboardshortcut.context.filebrowser') },
        'global' : { displayName : AJS.I18n.getText('stash.web.keyboardshortcut.context.global') },
        'pull-request' : { displayName : AJS.I18n.getText('stash.web.keyboardshortcut.context.pull-request') },
        'pull-request-list' : { displayName : AJS.I18n.getText('stash.web.keyboardshortcut.context.pull-request-list') },
        'pull-request-overview' : { displayName : AJS.I18n.getText('stash.web.keyboardshortcut.context.pull-request') },
        'sourceview' : { displayName : AJS.I18n.getText('stash.web.keyboardshortcut.context.sourceview') }
    };

    var keyboardShortcuts = new KeyboardShortcuts();
    exports.onReady = function() {
        // hardcoded keyboard link selector for now

        keyboardShortcuts._bind($('.keyboard-shortcut-link'));

        var onAfterDocumentReady = $.Deferred();
        $(document).ready(function() { // ensure everyone has had a chance to bind listeners before initializing
            setTimeout(function() {
                onAfterDocumentReady.resolve();
            }, 0);
        });

        AJS.bind("register-contexts.keyboardshortcuts", function(e, data) {

            keyboardShortcuts._setRegistry(data.shortcutRegistry);
            keyboardShortcuts.enableContext("global");

            onAfterDocumentReady.done(function() {
                events.trigger('stash.widget.keyboard-shortcuts.register-contexts', keyboardShortcuts, keyboardShortcuts);
            });
        });

        AJS.bind("shortcuts-loaded.keyboardshortcuts", function(e, data) {
            KeyboardShortcuts.internalizeShortcuts(data.shortcuts);
        });

        // TODO: load real keyboard shortcuts version.  Updating keyboard shortcuts will cause caching hell currently.
        AJS.params["keyboardshortcut-hash"] = 'bundled';

        AJS.trigger("initialize.keyboardshortcuts");
    };

    /**
     * Converts 'ctrl+shift+p' to ' Type (Ctrl + Shift + p)' (or the version for Mac)
     * and appends it to $el's title attribute.
     */
    exports.addTooltip = function($el, keys) {
        var keysTitle = _(keys.split('+')).chain().map(KeyboardShortcuts.convertOSModifier).map(function(key) {
            if (key === 'shift') {
                return AJS.I18n.getText('stash.web.keyboardshortcut.key.shift');
            } else if (key === 'ctrl') {
                return KeyboardShortcuts.convertOSModifier(AJS.I18n.getText('stash.web.keyboardshortcut.key.ctrl'));
            } else {
                return key;
            }
        }).value().join(' + ');
        var oldTitle = $el.attr('title');
        $el.attr('title', oldTitle + AJS.I18n.getText('stash.web.keyboardshortcut.type', keysTitle));
        return {
            remove: function() {
                $el.attr('title', oldTitle);
            }
        };
    };

    exports.showDialog = function() {
        if (keyboardShortcuts) {
            keyboardShortcuts._show();
        }
    };

    exports.resetContexts = function() {
        keyboardShortcuts.resetContexts();
    };
});
