/**
 * This module contains a dynamic set of properties based on the current page.
 * Relevant model data from the page will be exposed here for your use.
 *
 * **Web Resource:** com.atlassian.stash.stash-web-api:state
 *
 * @namespace stash/api/util/state
 */
define('stash/api/util/state', [
    'jquery',
    'underscore',
    'model/page-state'
],
/**
 * @exports stash/api/util/state
 */
function(
    $,
    _,
    pageState
) {

    "use strict";

    /**
     * Available on every page. Returns information about the current user, if there is a logged in user.
     *
     * @function
     * @name getCurrentUser
     * @memberof stash/api/util/state
     * @returns {JSON.StashUserJSON} An object describing the current user, if they are logged in.
     */

    /**
     * Available on every page. Returns information about the currently viewed project, if there is one being viewed.
     *
     * @function
     * @name getProject
     * @memberof stash/api/util/state
     * @returns {JSON.ProjectJSON} An object describing the current project, if there is one.
     */

    /**
     * Available on every page. Returns information about the currently viewed pull request, if there is one being viewed.
     *
     * @function
     * @name getPullRequest
     * @memberof stash/api/util/state
     * @returns {JSON.PullRequestJSON} An object describing the current pull request, if there is one.
     */

    /**
     * Available on every page. Returns information about the currently viewed repository, if there is one being viewed.
     *
     * @function
     * @name getRepository
     * @memberof stash/api/util/state
     * @returns {JSON.RepositoryJSON} An object describing the current repository, if there is one.
     */

    /**
     * Available on every page. Returns information about the currently viewed ref (e.g. currently selected branch), if there is one being viewed.
     *
     * @function
     * @name getRef
     * @memberof stash/api/util/state
     * @returns {JSON.RefJSON} An object describing the current ref, if there is one.
     */

    /**
     * Available on pages that show a file (Pull Request, Commit, File Browser).
     * Returns path information about the currently viewed file.
     *
     * @function
     * @name getFilePath
     * @memberof stash/api/util/state
     * @returns {JSON.PathJSON} An object describing the current file, if there is one.
     */


    var readablePageState = {};

    // getters to expose from the internal page state.
    var exposed = ['getCurrentUser', 'getProject', 'getRepository', 'getPullRequest', 'getRevisionRef', 'getFilePath', 'getChangeset'];

    function add(method, prop) {
        if (typeof method === 'function' && _.indexOf(exposed, prop) !== -1) {
            var getter = pageState[prop];
            prop = { // rename some properties for public consumption
                'getRevisionRef' : 'getRef',
                'getChangeset' : 'getCommit'
            }[prop] || prop;
            readablePageState[prop] = function() {
                var val = getter.call(pageState);
                if (val == null) {
                    // We erroneously returned an empty object before in this case. That sucks.
                    // We'd like to return the null or undefined value, but if people are working around this like
                    // by checking state.getPullRequest().id, those checks would start to throw.
                    // So this middle ground lets both work:
                    // if (state.getPullRequest()) will be false, and if(state.getPullRequest().id will return undefined)
                    // This is technically a breaking change, but the benefit is significant for everyone using the API.
                    // The current behavior is a bug.
                    // DEPRECATED in 3.2, switch to returning `val` directly in 4.0
                    return false;
                }
                if (typeof val !== 'object') {
                    return val;
                }
                if (val.toJSON) {
                    val = val.toJSON();
                }
                return $.extend(true, {}, val);
            };
        }
    }

    pageState.extend = (function (oldExtend) {
        return function() {
            var ret = oldExtend.apply(this, arguments);
            _.each(pageState, add);
            return ret;
        };
    })(pageState.extend);

    _.each(pageState, add);

    return readablePageState;
});