/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.stash.internal.backup;

import com.atlassian.stash.exception.ArgumentValidationException;
import com.atlassian.stash.exception.NoSuchEntityException;
import com.atlassian.stash.i18n.I18nService;
import com.atlassian.stash.internal.ApplicationSettings;
import com.atlassian.stash.internal.annotation.Unsecured;
import com.atlassian.stash.internal.backup.Backup;
import com.atlassian.stash.internal.backup.BackupException;
import com.atlassian.stash.internal.backup.BackupFeature;
import com.atlassian.stash.internal.backup.BackupFeatureMode;
import com.atlassian.stash.internal.backup.BackupFeatures;
import com.atlassian.stash.internal.backup.BackupService;
import com.atlassian.stash.internal.backup.FileBackup;
import com.atlassian.stash.internal.backup.SimpleBackupFeature;
import com.atlassian.stash.internal.backup.liquibase.LiquibaseDao;
import com.atlassian.stash.internal.maintenance.BaseMaintenanceCompletionCallback;
import com.atlassian.stash.internal.maintenance.MaintenanceCompletionCallback;
import com.atlassian.stash.internal.maintenance.MaintenanceService;
import com.atlassian.stash.internal.maintenance.MaintenanceTask;
import com.atlassian.stash.internal.maintenance.MaintenanceTaskFactory;
import com.atlassian.stash.internal.maintenance.MaintenanceTaskMonitor;
import com.atlassian.stash.internal.maintenance.MaintenanceType;
import com.atlassian.stash.internal.maintenance.backup.BackupClientProgressCallback;
import com.atlassian.stash.internal.maintenance.backup.BackupTask;
import com.atlassian.stash.util.Page;
import com.atlassian.stash.util.PageRequest;
import com.atlassian.stash.util.PageUtils;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Collections2;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import java.io.File;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.regex.Pattern;
import javax.annotation.Nonnull;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.filefilter.FalseFileFilter;
import org.apache.commons.io.filefilter.IOFileFilter;
import org.apache.commons.io.filefilter.RegexFileFilter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.stereotype.Service;

@Service(value="backupService")
public class DefaultBackupService
implements BackupService {
    private static final String EXTENSION_ZIP = ".zip";
    private static final int LENGTH_SUFFIX = (new SimpleDateFormat("yyyyMMdd-HHmmss-SSS'Z'").format(new Date()) + ".zip").length();
    private static final int LENGTH_EXTENSION = ".zip".length();
    private static final Pattern PATTERN_UTC_FILE_NAME = Pattern.compile("backup-[^\\\\/\\.]+-([0-9]{8}-[0-9]{6}-[0-9]{3})Z\\.zip");
    private static final Comparator<File> FILE_NAME_TIMESTAMP_COMPARATOR = new Comparator<File>(){

        @Override
        public int compare(File left, File right) {
            return this.fileNameTimestamp(right).compareTo(this.fileNameTimestamp(left));
        }

        private String fileNameTimestamp(File file) {
            int length = file.getName().length();
            return file.getName().substring(length - LENGTH_SUFFIX, length - LENGTH_EXTENSION);
        }
    };
    private static final Function<Class<?>, BackupFeature> CLASS_TO_BACKUP_FEATURE = new Function<Class<?>, BackupFeature>(){

        public BackupFeature apply(Class<?> input) {
            return new SimpleBackupFeature("database", input.getSimpleName(), BackupFeatureMode.RESTORE);
        }
    };
    private static final Logger log = LoggerFactory.getLogger(DefaultBackupService.class);
    private final I18nService i18nService;
    private final MaintenanceService maintenanceService;
    private final MaintenanceTaskFactory maintenanceTaskFactory;
    private final ApplicationSettings settings;
    private final LiquibaseDao liquibaseDao;
    private volatile BackupClientProgressCallback clientProgressCallback;

    @Autowired
    public DefaultBackupService(I18nService i18nService, MaintenanceService maintenanceService, MaintenanceTaskFactory maintenanceTaskFactory, ApplicationSettings settings, LiquibaseDao liquibaseDao) {
        this.i18nService = i18nService;
        this.maintenanceService = maintenanceService;
        this.maintenanceTaskFactory = maintenanceTaskFactory;
        this.settings = settings;
        this.liquibaseDao = liquibaseDao;
    }

    @Nonnull
    @PreAuthorize(value="hasGlobalPermission('SYS_ADMIN')")
    public MaintenanceTaskMonitor backup() {
        try {
            BackupTask backupTask = this.maintenanceTaskFactory.backupTask();
            MaintenanceTaskMonitor taskMonitor = this.maintenanceService.start((MaintenanceTask)backupTask, MaintenanceType.BACKUP);
            this.clientProgressCallback = backupTask.getClientProgressCallback();
            taskMonitor.registerCallback((MaintenanceCompletionCallback)new BaseMaintenanceCompletionCallback(){

                @Override
                protected void onCompletion() {
                    DefaultBackupService.this.clientProgressCallback = null;
                }
            });
            return taskMonitor;
        }
        catch (IllegalStateException e) {
            log.error("An attempt to start a backup was blocked because maintenance is already in progress");
            throw new BackupException(this.i18nService.getKeyedText("stash.backup.already.running", "Backup cannot be started because other system maintenance is already in progress.", new Object[0]));
        }
    }

    @PreAuthorize(value="hasGlobalPermission('SYS_ADMIN')")
    public boolean delete(@Nonnull Backup backup) {
        String name = this.validateName(((Backup)Preconditions.checkNotNull((Object)backup, (Object)"backup")).getName());
        File file = new File(this.settings.getBackupDir(), name);
        return file.isFile() && file.delete();
    }

    @Nonnull
    @PreAuthorize(value="hasGlobalPermission('SYS_ADMIN')")
    public Page<Backup> findAll(@Nonnull PageRequest pageRequest) {
        Preconditions.checkNotNull((Object)pageRequest, (Object)"pageRequest");
        List<File> files = this.listBackupFiles();
        if (pageRequest.getStart() > files.size()) {
            return PageUtils.createEmptyPage((PageRequest)pageRequest);
        }
        Iterable<File> page = pageRequest.getStart() > 0 ? Iterables.skip(files, (int)pageRequest.getStart()) : files;
        return PageUtils.createPage((Iterable)Iterables.transform(page, FileBackup.FILE_TRANSFORM), (PageRequest)pageRequest);
    }

    @PreAuthorize(value="hasGlobalPermission('SYS_ADMIN')")
    public Backup findByName(@Nonnull String name) {
        name = this.validateName(name);
        File backup = new File(this.settings.getBackupDir(), name);
        if (backup.isFile()) {
            return new FileBackup(backup);
        }
        return null;
    }

    @Nonnull
    @PreAuthorize(value="hasGlobalPermission('SYS_ADMIN')")
    public Backup getByName(@Nonnull String name) {
        Backup backup = this.findByName(name);
        if (backup == null) {
            throw new NoSuchEntityException(this.i18nService.getKeyedText("stash.service.backup.nosuchbackup", "No backup exists with name {0}", new Object[]{name}));
        }
        return backup;
    }

    @Nonnull
    @PreAuthorize(value="hasGlobalPermission('SYS_ADMIN')")
    public Backup getLatest() {
        List<File> files = this.listBackupFiles();
        if (files.isEmpty()) {
            throw new NoSuchEntityException(this.i18nService.getKeyedText("stash.service.backup.nobackups", "The system has not created any backups.", new Object[0]));
        }
        return new FileBackup(files.get(0));
    }

    @Nonnull
    @PreAuthorize(value="hasGlobalPermission('SYS_ADMIN')")
    public List<BackupFeature> getFeatures() {
        return ImmutableList.builder().addAll(BackupFeatures.getFeatures()).addAll((Iterable)Collections2.transform((Collection)this.liquibaseDao.findCustomChanges(), CLASS_TO_BACKUP_FEATURE)).build();
    }

    @Unsecured(value="Updating the client's backup progress cannot be secured; the database may not be available")
    public void updateClientProgress(int percentage) {
        Preconditions.checkArgument((percentage >= 0 && percentage <= 100 ? 1 : 0) != 0, (Object)"Progress must be between 0 and 100");
        BackupClientProgressCallback listener = this.clientProgressCallback;
        if (listener == null) {
            throw new NoSuchEntityException(this.i18nService.getKeyedText("stash.service.backup.not.backing.up", "The system is not currently backing up", new Object[0]));
        }
        listener.onProgressUpdate(percentage);
    }

    private List<File> listBackupFiles() {
        ArrayList files = Lists.newArrayList((Iterable)FileUtils.listFiles((File)this.settings.getBackupDir(), (IOFileFilter)new RegexFileFilter(PATTERN_UTC_FILE_NAME), (IOFileFilter)FalseFileFilter.INSTANCE));
        Collections.sort(files, FILE_NAME_TIMESTAMP_COMPARATOR);
        return files;
    }

    private String validateName(String name) {
        Preconditions.checkNotNull((Object)name, (Object)"name");
        name = name.trim();
        if (!PATTERN_UTC_FILE_NAME.matcher(name).matches()) {
            throw new ArgumentValidationException(this.i18nService.getKeyedText("stash.service.backup.invalidname", "{0} is not a valid backup name.", new Object[]{name}));
        }
        return name;
    }
}

