define('feature/branch/branch-copy', [
    'aui',
    'aui/flag',
    'jquery',
    'lodash',
    'zero-clipboard',
    'stash/api/util/events',
    'util/navigator',
    'widget/quick-copy-text',
    'exports'
], function(
    AJS,
    auiFlag,
    $,
    _,
    ZeroClipboard,
    events,
    navigatorUtil,
    quickCopyText,
    exports
) {

    'use strict';

    quickCopyText.onReady();
    // we don't want ZeroConfig to bubble the click events up to the dropdown2 menu item, which has the click
    // handler to show the manual copy dialog. Needed for accessibility and keyboard trigger of the dropdown2
    // item
    ZeroClipboard.config({
        forceHandCursor: true,
        bubbleEvents: false
    });

    /**
     * Displays a AUI Dialog2 with the branch name in a text field as a fallback if Flash is unavailable
     *
     * @param {string} branchName the branch/ref name
     */
    function showManualCopyDialog(branchName) {
        var dialog = AJS.dialog2(aui.dialog.dialog2({
            titleText: AJS.I18n.getText('stash.branch.copy.dialog.title'),
            content: stash.feature.branchCopy.dialogContent({
                branchName: branchName,
                flashUnsupported: ZeroClipboard.isFlashUnusable()
            }),
            footerHintText: AJS.I18n.getText('stash.branch.copy.dialog.tip', navigatorUtil.isMac() ? '⌘ + c' : 'Ctrl + c'),
            removeOnHide: true
        })).show();
        dialog.$el.find('form').on('submit', function(e) { e.preventDefault(); });
        dialog.$el.find('#branch-copy-field').on('copy', function() {
            _.defer(function () {
                dialog.hide();
            });
            auiFlag({
                type: 'success',
                title: AJS.I18n.getText('stash.branch.copy.copied'),
                close: 'auto'
            });
        }).focus().select();
    }

    function manuallyHideZeroclipboardFlash() {
        /* Hack: In Firefox, when clicking on the copy branch dropdown item, the flash element is still clickable
           even though the dropdown has hidden. Manually hide zeroclipboard flash element.
           https://jira.atlassian.com/browse/STASHDEV-5262 */
        $('#global-zeroclipboard-html-bridge').css({
            top: '-9999px',
            left : '-9999px'
        });
    }

    /**
     * Binds the branch copy functionality to a AUI dropdown2 menu item
     *
     * @param {string} linkSelector Selector for the menu item that triggers the branch name copy
     * @param {string} dropDownSelector Selector for the Dropdown2
     * @param {Function} getBranchNameFunction Function that returns the branch or ref name
     * @param {string} context - 'list' or 'selector' to denote the source of the dropdown link,
     *                           either the branch list or a branch selector
     * @this {ZeroClipboard} the ZeroClipboard client
     */
    function bindToDropdownLink(linkSelector, dropDownSelector, getBranchNameFunction, context) {
        var branchName;
        var manualCopyClickHandler = function(e) {
            e.preventDefault();
            showManualCopyDialog(branchName);
        };

        //selecting on the document as the drop down is absolutely positioned and may not have a parent other than the document
        $(document).on('aui-dropdown2-show', dropDownSelector, function () {
            var $dropdownMenu = $(this);
            var $copyTrigger = $(linkSelector);
            branchName = getBranchNameFunction($dropdownMenu);
            var clipboard;

            var afterCopy = function(e) {
                // Can't do it on aui-dropdown2-hide event because of ordering issues
                auiFlag({
                    type: 'success',
                    title: AJS.I18n.getText('stash.branch.copy.copied'),
                    close: 'auto'
                });
                events.trigger('stash.feature.branch-copy.branchNameCopied', null, {
                    name: e.data["text/plain"],
                    context: context,
                    webItemKey: $(this).data('web-item-key')
                });
                $dropdownMenu[0].hide();
                clipboard.destroy();
                ZeroClipboard.off('aftercopy', afterCopy);
            };

            if (!ZeroClipboard.isFlashUnusable()) {
                clipboard = new ZeroClipboard($copyTrigger);
                clipboard.on('copy', function () {
                    this.setText(branchName);
                });
                ZeroClipboard.on('aftercopy', afterCopy);
            }

            $copyTrigger.one('click', manualCopyClickHandler);
            $dropdownMenu.on('aui-dropdown2-hide', function() {
                $copyTrigger.off('click', manualCopyClickHandler);
                manuallyHideZeroclipboardFlash();
                if (clipboard) {
                    clipboard.destroy();
                }
            });
        });
    }

    exports.bindToDropdownLink = bindToDropdownLink;
});