package com.atlassian.seraph.config;

import com.atlassian.seraph.util.RedirectUtils;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

public class DefaultRedirectPolicy implements RedirectPolicy
{
    private boolean allowAnyUrl = false;

    @Override
    public void init(final Map<String, String> params, final SecurityConfig config)
    {
        // params is never allowed to be null.
        if (params == null)
        {
            throw new IllegalArgumentException("params is not allowed to be null");
        }
        // Allow applications to set allow.any.redirect.url=true for legacy behaviour
        allowAnyUrl = "true".equals(params.get("allow.any.redirect.url"));
    }

    /**
     * Returns true if we allow redirect to any URL at all.
     * By default this is false, however it may be configured to true in the Seraph config file to allow legacy behaviour.
     *
     * @return true if we allow redirect to any URL at all.
     */
    public boolean isAllowAnyUrl()
    {
        return allowAnyUrl;
    }

    /**
     * Checks if the given redirectURL is permitted.
     * <p/>
     * Uses the configured redirect rules to see if we are allowed to redirect to the given URL.
     * By default, the following is allowed:
     * <ul>
     *   <li>Any relative URL</li>
     *   <li>An absolute URL to the same context path as the current incoming request</li>
     * </ul>
     * You can configure this "security-policy" in the Seraph XML config file.
     * eg:
     * <pre>
     *  &lt;redirect-policy class="com.atlassian.seraph.config.SimpleRedirectPolicy"&gt;
     *    &lt;init-param&gt;
     *      &lt;param-name&gt;allow.any.redirect.url&lt;/param-name&gt;
     *      &lt;param-value&gt;true&lt;/param-value&gt;
     *    &lt;/init-param&gt;
     *  &lt;/redirect-policy&gt;
     * </pre>
     * <p/>
     *
     * @param redirectUrl Requested redirect URL to be verified.
     * @param request The current incoming request.
     * @return <code>true</code> if this redirectURL is allowed.
     */
    @Override
    public boolean allowedRedirectDestination(final String redirectUrl, final HttpServletRequest request)
    {
        // Test for total trust
        if (allowAnyUrl)
        {
            return true;
        }
        // Otherwise we use default behaviour: allow valid redirects to the same context.
        URI uri;
        try
        {
            // Attempt to parse the URI
            uri = new URI(redirectUrl);
        }
        catch (final URISyntaxException e)
        {
            // Invalid URI - not allowed. This stops possible header injection attacks (see SER-127)
            // but it is also good in general that if we can't parse a URI, then we can't trust it.
            return false;
        }
        catch (final IllegalArgumentException e)
        {
            return false;
        }
        // The URI is valid - if it is absolute, then check that it is to the same context
        return uri.getHost() == null || RedirectUtils.sameContext(redirectUrl, request);
    }
}
