package com.atlassian.seraph.util;

import org.apache.commons.codec.binary.Base64;
import org.apache.log4j.Category;

import javax.servlet.ServletContext;
import javax.servlet.ServletRequest;
import javax.servlet.http.HttpServletRequest;

import com.atlassian.seraph.config.SecurityConfig;
import com.atlassian.seraph.auth.Authenticator;

public class SecurityUtils
{
    private static final Category log = Category.getInstance(SecurityUtils.class);

    /** The Basic authorization encoding method. Includes trailing space, for convenient startsWith() checks, substring() and concatenation. */
    private static final String BASIC_AUTHZ_TYPE_PREFIX = "Basic ";

    private static final String ALREADY_FILTERED = "loginfilter.already.filtered";

    public static Authenticator getAuthenticator(ServletContext servletContext)
    {
        SecurityConfig securityConfig = (SecurityConfig) servletContext.getAttribute(SecurityConfig.STORAGE_KEY);

        if (securityConfig.getAuthenticator() == null)
        {
            log.error("ack! Authenticator is null!!!");
        }

        return securityConfig.getAuthenticator();
    }

    public static boolean isBasicAuthorizationHeader(String header)
    {
        return (header != null) && header.startsWith(BASIC_AUTHZ_TYPE_PREFIX);
    }

    /**
     * Extracts the username and password from the given header string (including the 'Basic ' prefix).
     * @param basicAuthorizationHeader the header to decode.
     * @return the credentials, or a username and password of "" if there were no credentials to decode.
     */
    public static UserPassCredentials decodeBasicAuthorizationCredentials(String basicAuthorizationHeader)
    {
        final String base64Token = basicAuthorizationHeader.substring(BASIC_AUTHZ_TYPE_PREFIX.length());
        // note: header fields are not UTF-8, but ISO Latin 1, even when base 64 encoded
        final String token = SecurityUtils.decodeBase64(base64Token, CharsetUtils.ISO_LATIN_1_CHARSET);

        String userName = "";
        String password = "";

        final int delim = token.indexOf(":");

        if (delim != -1)
        {
            userName = token.substring(0, delim);
            password = token.substring(delim + 1);
        }
        return new UserPassCredentials(userName, password);
    }

    /**
     * Reverses the operation of decodeBasicAuthorizationCredentials. Mainly for unit tests, or ServletFilters faking
     * basic authorization.
     * @param username the username to encode.
     * @param password the password to encode.
     * @return the encoded credentials.
     */
    public static String encodeBasicAuthorizationCredentials(String username, String password)
    {
        return BASIC_AUTHZ_TYPE_PREFIX + encodeBase64(username + ":" + password, CharsetUtils.ISO_LATIN_1_CHARSET);

    }

    /**
     * Disables seraph filtering
     * @param request
     * @since 2.5
     */
    public static void disableSeraphFiltering(ServletRequest request)
    {
        request.setAttribute(ALREADY_FILTERED, true);
    }

    /**
     * Checks if Seraph filtering is disabled
     * @param request
     * @return disabled
     * @since 2.5
     */
    public static boolean isSeraphFilteringDisabled(ServletRequest request)
    {
        return request.getAttribute(ALREADY_FILTERED) != null;
    }

    /**
     * Decodes the given Base64 encoded string using the given charset.
     * @param encodedString the string to decode.
     * @param charset the charset to decode it with.
     * @return the decoded string.
     */
    private static String decodeBase64(String encodedString, String charset)
    {
        byte[] encodedBytes = CharsetUtils.bytesFromString(encodedString, CharsetUtils.US_ASCII_CHARSET);
        byte[] decodedBytes = Base64.decodeBase64(encodedBytes);
        return CharsetUtils.stringFromBytes(decodedBytes, charset);
    }

    private static String encodeBase64(String decodedString, String charset)
    {
        byte[] decodedBytes = CharsetUtils.bytesFromString(decodedString, charset);
        byte[] encodedBytes = Base64.encodeBase64(decodedBytes);
        return CharsetUtils.stringFromBytes(encodedBytes, CharsetUtils.US_ASCII_CHARSET);
    }

    /**
     * User credentials including a username and a password.
     * TODO Eliminate duplication between this class and {@link com.atlassian.seraph.filter.PasswordBasedLoginFilter.UserPasswordPair}
     */
    public static class UserPassCredentials
    {
        private final String username;
        private final String password;

        public UserPassCredentials(String username, String password)
        {
            this.username = username;
            this.password = password;
        }

        public String getUsername()
        {
            return username;
        }

        public String getPassword()
        {
            return password;
        }
    }
}
