package com.atlassian.seraph.util;

import edu.emory.mathcs.backport.java.util.concurrent.locks.ReadWriteLock;
import edu.emory.mathcs.backport.java.util.concurrent.locks.ReentrantReadWriteLock;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map;

/**
 * Caches the results of the {@link PathMapper}
 */
public class CachedPathMapper implements IPathMapper
{
    private final ReadWriteLock lock = new ReentrantReadWriteLock();
    private final PathMapper delegate = new PathMapper();
    private final Map cacheMap;
    private final Map cacheAllMap;

    /**
     * Creates a CachedPathMapper object that will use cacheMap to cache the results of the {@link #get(String)} calls and cacheAllMap to cache the
     * results of the {@link #getAll(String)} class.
     * <p>
     * Use the passed in maps for caches. The maps must be thread-safe as far as {@link Map#get(Object)} calls are concerned as gets may happen
     * concurrently. An access ordered map should be wrapped in a synchronizedMap wrapper.
     * 
     * @param cacheMap
     *            for caching results of {@link #get(String)} calls
     * @param cacheAllMap
     *            for caching results of {@link #getAll(String)} calls
     */
    public CachedPathMapper(final Map cacheMap, final Map cacheAllMap)
    {
        this.cacheMap = cacheMap;
        this.cacheAllMap = cacheAllMap;
    }

    public String get(final String path)
    {
        lock.readLock().lock();
        try
        {
            // Check the cache
            final String result = (String) cacheMap.get(path);
            if (result != null)
            {
                return result;
            }
        }
        finally
        {
            lock.readLock().unlock();
        }
        return getWithLock(path);
    }

    private String getWithLock(final String path)
    {
        final String result = delegate.get(path);
        // Get the result from PathMapper
        // Cache the result
        lock.writeLock().lock();
        try
        {
            cacheMap.put(path, result);
            return result;
        }
        finally
        {
            lock.writeLock().unlock();
        }
    }

    public Collection getAll(final String path)
    {
        lock.readLock().lock();
        try
        {
            final Collection result = (Collection) cacheAllMap.get(path);
            // Check the cache
            if (result != null)
            {
                // The result for this key is cached, return the value
                return result;
            }
        }
        finally
        {
            lock.readLock().unlock();
        }
        return getAllWithLock(path);
    }

    private Collection getAllWithLock(final String path)
    {
        // Get the result from delegate PathMapper
        final Collection result = delegate.getAll(path);
        // Cache the result
        lock.writeLock().lock();
        try
        {
            cacheAllMap.put(path, result);
            return result;
        }
        finally
        {
            lock.writeLock().unlock();
        }
    }

    public void set(final Map/* <String, String> */patterns)
    {
        lock.writeLock().lock();
        try
        {
            for (final Iterator/* <Entry<String, String>> */it = patterns.entrySet().iterator(); it.hasNext();)
            {
                final Map.Entry/* <String, String> */entry = (Map.Entry) it.next();
                put((String) entry.getKey(), (String) entry.getValue());
            }
        }
        finally
        {
            lock.writeLock().unlock();
        }
    }

    public void put(final String key, final String pattern)
    {
        lock.writeLock().lock();
        try
        {
            // Let the delegate mapper update the patterns
            delegate.put(key, pattern);
            cacheMap.remove(key);
            cacheAllMap.remove(key);
        }
        finally
        {
            lock.writeLock().unlock();
        }
    }

    public String toString()
    {
        return delegate.toString();
    }
}