package com.atlassian.seraph.filter;

import java.util.StringTokenizer;

import javax.servlet.http.HttpServletRequest;

import com.opensymphony.user.provider.ejb.util.Base64;

/**
 * This is a filter that logs the user in. It parses a standard HTTP based authentication requst and logs the user in.
 * At the moment it supports only a BASIC authentication scheme. The simple way of using it manually is to request a URL in the following form:
 * protocol://user:password@host[:port]/path
 * e.g.:
 * https://juancho:sillyPass@bamboo.developer.atlassian.com/
 * <p>
 * If authentication is successful, the user will be redirected by the filter to the URL given
 * by the session attribute at SecurityFilter.ORIGINAL_URL_KEY.
 * <p>
 * If this URL doesn't exist, it will look for a parameter 'os_destination' to use as the redirected URL instead.
 * <p>
 * If neither is found, it is assumed that the page will check the authorisation status and handle redirection itself.
 * <p>
 * From the any other filter in the request, or the servlet/JSP/action which processes the request, you can look up the
 * status of the authorisation attempt. The status is a String request attribute, with the key 'os_authstatus'.
 * <p>
 * The possible statuses are:
 * <ul>
 *  <li> LoginFilter.LOGIN_SUCCESS - the login was processed, and user was logged in
 *  <li> LoginFilter.LOGIN_FAILURE - the login was processed, the user gave a bad username or password
 *  <li> LoginFilter.LOGIN_ERROR - the login was processed, an exception occurred trying to log the user in
 *  <li> LoginFilter.LOGIN_NOATTEMPT - the login was no processed, no form parameters existed
 * </ul>
 */
public class HttpAuthFilter extends PasswordBasedLoginFilter
{
	UserPasswordPair extractUserPasswordPair(HttpServletRequest request)
	{
        String auth = request.getHeader("Authorization");
		if (auth != null)
		{
			if (auth.toUpperCase().startsWith("BASIC "))
			{
				// Get encoded user and password, comes after "BASIC "
				String userpassEncoded = auth.substring("BASIC ".length());

				// Decode it, using any base 64 decoder
				byte[] decodedBytes = Base64.decode(userpassEncoded.getBytes());
				String userpassDecoded = new String(decodedBytes);

				if (userpassDecoded.length() != 0 && (userpassDecoded.indexOf(':') != -1))
				{
					StringTokenizer tokenizer = new StringTokenizer(userpassDecoded, ":");
					String username = tokenizer.nextToken();
					String password = tokenizer.nextToken();
					return new UserPasswordPair(username, password, false);
				}
			}
		}
		return null;
	}
}
