package com.atlassian.seraph.filter;

import com.atlassian.seraph.RequestParameterConstants;
import com.atlassian.seraph.auth.Authenticator;
import com.atlassian.seraph.config.SecurityConfig;
import com.atlassian.seraph.util.RedirectUtils;
import org.apache.log4j.Logger;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.Principal;
import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import javax.servlet.http.HttpServletResponse;

/**
 * This is a base authentication filter. It delegates the actual login process to a child class but takes care of the redirection process.
 * <p/>
 * If the authentication is successful, the user will be redirected by the filter to the URL given
 * by the session attribute at SecurityFilter.ORIGINAL_URL_KEY.
 * <p/>
 * If this URL doesn't exist, it will look for a parameter 'os_destination' to use as the redirected URL instead.
 * <p/>
 * If neither is found, it is assumed that the page will check the authorisation status and handle redirection itself.
 * <p/>
 * From the any other filter in the request, or the servlet/JSP/action which processes the request, you can look up the
 * status of the authorisation attempt. The status is a String request attribute, with the key 'os_authstatus'.
 * <p/>
 * The possible statuses are:
 * <ul>
 * <li> LoginFilter.LOGIN_SUCCESS - the login was processed, and user was logged in
 * <li> LoginFilter.LOGIN_FAILURE - the login was processed, the user gave a bad username or password
 * <li> LoginFilter.LOGIN_ERROR - the login was processed, an exception occurred trying to log the user in
 * <li> LoginFilter.LOGIN_NOATTEMPT - the login was no processed, no form parameters existed
 * </ul>
 */
public abstract class BaseLoginFilter implements Filter
{
    private static final Logger log = Logger.getLogger(BaseLoginFilter.class);

    private FilterConfig filterConfig = null;
    protected static final String ALREADY_FILTERED = "loginfilter.already.filtered";
    public static final String LOGIN_SUCCESS = "success";
    public static final String LOGIN_FAILED = "failed";
    public static final String LOGIN_ERROR = "error";
    public static final String LOGIN_NOATTEMPT = null;
    public static final String OS_AUTHSTATUS_KEY = "os_authstatus";
    private SecurityConfig securityConfig = null;

    public BaseLoginFilter()
    {
        super();
    }

    public void init(FilterConfig config)
    {
        // log.debug("LoginFilter.init");
        this.filterConfig = config;
    }

    public void destroy()
    {
        // log.debug("LoginFilter.destroy");
        filterConfig = null;
    }

    /**
     * @deprecated Not needed in latest version of Servlet 2.3 API
     */
    public FilterConfig getFilterConfig()
    {
        return filterConfig;
    }

    /**
     * @deprecated Not needed in latest version of Servlet 2.3 API - replaced by init().
     */
    public void setFilterConfig(FilterConfig filterConfig)
    {
        if (filterConfig != null) //it seems that Orion 1.5.2 calls this with a null config.
        {
            init(filterConfig);
        }
    }

    public void doFilter(ServletRequest req, ServletResponse res, FilterChain chain)
            throws IOException, ServletException
    {
        final boolean dbg = log.isDebugEnabled();
        // wrap the request with one that returns the User as the Principal
        req = new SecurityHttpRequestWrapper((HttpServletRequest) req);

        if (req.getAttribute(ALREADY_FILTERED) == null && getSecurityConfig().getController().isSecurityEnabled())
        {
            req.setAttribute(ALREADY_FILTERED, Boolean.TRUE);
            req.setAttribute(OS_AUTHSTATUS_KEY, LOGIN_NOATTEMPT);

            HttpServletRequest request = (HttpServletRequest) req;
            HttpServletResponse response = (HttpServletResponse) res;

            if (dbg)
            {
                String url = request.getServletPath() +
                             (request.getPathInfo() == null ? "" : request.getPathInfo()) +
                             (request.getQueryString() == null ? "" : "?" + request.getQueryString());
                log.debug("____ Attempting login for : '" + url + "'");
            }

            String status = login(request, response);
            request.setAttribute(OS_AUTHSTATUS_KEY, status);
            if (dbg)
            {
                log.debug("Login completed - set " + OS_AUTHSTATUS_KEY + " attribute to '" + status + "'");
            }

            // if we successfully logged in - look for an original URL to forward to
            if (status == LOGIN_SUCCESS && redirectToOriginalDestination(request, response))
            {
                return;
            }
        }

        chain.doFilter(req, res);
    }

    /**
     * Performs the actual authentication (if required) and returns the status code. Status code is chosen to be one of these:
     * <p/>
     * The possible statuses are:
     * <ul>
     * <li> BaseLoginFilter.LOGIN_SUCCESS - the login was processed, and user was logged in
     * <li> BaseLoginFilter.LOGIN_FAILURE - the login was processed, the user gave a bad username or password
     * <li> BaseLoginFilter.LOGIN_ERROR - the login was processed, an exception occurred trying to log the user in
     * <li> BaseLoginFilter.LOGIN_NOATTEMPT - the login was no processed, no form parameters existed
     * </ul>
     *
     * @param request
     * @param response
     * @return authentication status
     */
    public abstract String login(HttpServletRequest request, HttpServletResponse response);

    private class SecurityHttpRequestWrapper extends HttpServletRequestWrapper
    {
        private HttpServletRequest request;

        public SecurityHttpRequestWrapper(HttpServletRequest request)
        {
            super(request);
            this.request = request;
        }

        public String getRemoteUser()
        {
            Principal user = getUserPrincipal();
            return (user == null) ? null : user.getName();
        }

        public Principal getUserPrincipal()
        {
            return getSecurityConfig().getAuthenticator().getUser(request);
        }
    }

    /**
     * Redirect the response to the original destination if present
     *
     * @param request
     * @param response
     * @return true if a redirect was needed and issued
     * @throws IOException
     */
    protected boolean redirectToOriginalDestination(HttpServletRequest request, HttpServletResponse response)
            throws IOException
    {
        String originalURL = (String) request.getSession().getAttribute(getSecurityConfig().getOriginalURLKey());
        String destinationURL = request.getParameter(RequestParameterConstants.OS_DESTINATION);
        String redirectURL = null;

        if (originalURL != null)
        {
            request.getSession().setAttribute(getSecurityConfig().getOriginalURLKey(), null);
            redirectURL = originalURL;
        }
        else if (destinationURL != null)
        {
            redirectURL = destinationURL;
        }

        if (redirectURL == null)
        {
            return false;
        }

        if (!isAbsoluteUrl(redirectURL))
        {
            redirectURL = RedirectUtils.appendPathToContext(request.getContextPath(), redirectURL);
        }

        if (log.isDebugEnabled())
        {
            log.debug("Logged In - redirecting to: " + redirectURL);
        }

        response.sendRedirect(redirectURL);
        return true;
    }

    protected boolean isAbsoluteUrl(String url)
    {
        try
        {
            URI uri = new URI(url);
            return uri.isAbsolute();
        }
        catch (URISyntaxException e)
        {
            return false;
        }
    }

    protected Authenticator getAuthenticator()
    {
        return getSecurityConfig().getAuthenticator();
    }

    protected SecurityConfig getSecurityConfig()
    {
        if (securityConfig == null)
        {
            securityConfig = (SecurityConfig) filterConfig.getServletContext().getAttribute(SecurityConfig.STORAGE_KEY);
        }
        return securityConfig;
    }
}
